
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, LineChart, Line, PieChart, Pie, Cell, Area, AreaChart, ResponsiveContainer } from 'recharts';
import { TrendingUp, TrendingDown, Activity, Users, Droplets, Trash2, Zap, TreePine, Building, Award, Shield } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useAuth } from '@/components/Auth/AuthProvider';

export const RealTimeAnalytics = () => {
  const [dataPoints, setDataPoints] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [stats, setStats] = useState({
    totalDataPoints: 0,
    validated: 0,
    materialsTracked: 0,
    carbonFootprint: 0
  });
  const { user } = useAuth();

  useEffect(() => {
    if (user) {
      fetchAnalyticsData();
    }
  }, [user]);

  const fetchAnalyticsData = async () => {
    try {
      setLoading(true);
      const { data, error } = await supabase
        .from('esg_data_points')
        .select('*')
        .eq('user_id', user?.id);

      if (error) throw error;

      setDataPoints(data || []);
      
      // Calculate stats
      const total = data?.length || 0;
      const validated = data?.filter(d => d.validation_status === 'validated').length || 0;
      const materials = data?.filter(d => d.scope && d.scope !== 'not-applicable').length || 0;
      const carbonData = data?.filter(d => d.metric_name?.toLowerCase().includes('carbon') || d.metric_name?.toLowerCase().includes('emission')) || [];
      const totalCarbon = carbonData.reduce((sum, item) => {
        const value = typeof item.metric_value === 'number' ? item.metric_value : 0;
        return sum + value;
      }, 0);

      setStats({
        totalDataPoints: total,
        validated,
        materialsTracked: materials,
        carbonFootprint: totalCarbon
      });
    } catch (error) {
      console.error('Error fetching analytics data:', error);
    } finally {
      setLoading(false);
    }
  };

  // Group data for charts
  const monthlyData = dataPoints.reduce((acc: any[], item) => {
    const month = new Date(item.created_at).toLocaleDateString('en-US', { month: 'short', year: 'numeric' });
    const existing = acc.find(d => d.month === month);
    if (existing) {
      existing.dataPoints += 1;
      existing.emissions += typeof item.metric_value === 'number' ? item.metric_value : 0;
    } else {
      acc.push({
        month,
        dataPoints: 1,
        emissions: typeof item.metric_value === 'number' ? item.metric_value : 0
      });
    }
    return acc;
  }, []);

  const scopeData = [
    { name: 'Scope 1', value: dataPoints.filter(d => d.scope === 'Scope 1').length, color: '#ef4444' },
    { name: 'Scope 2', value: dataPoints.filter(d => d.scope === 'Scope 2').length, color: '#f97316' },
    { name: 'Scope 3', value: dataPoints.filter(d => d.scope === 'Scope 3').length, color: '#eab308' },
  ];

  const categoryData = [
    { name: 'Energy', value: dataPoints.filter(d => d.metric_name?.toLowerCase().includes('energy')).length },
    { name: 'Water', value: dataPoints.filter(d => d.metric_name?.toLowerCase().includes('water')).length },
    { name: 'Waste', value: dataPoints.filter(d => d.metric_name?.toLowerCase().includes('waste')).length },
    { name: 'Emissions', value: dataPoints.filter(d => d.metric_name?.toLowerCase().includes('emission') || d.metric_name?.toLowerCase().includes('carbon')).length },
    { name: 'Social', value: dataPoints.filter(d => d.metric_name?.toLowerCase().includes('employee') || d.metric_name?.toLowerCase().includes('diversity')).length },
    { name: 'Governance', value: dataPoints.filter(d => d.metric_name?.toLowerCase().includes('board') || d.metric_name?.toLowerCase().includes('governance')).length },
  ];

  if (loading) {
    return (
      <div className="flex items-center justify-center py-8">
        <div className="text-gray-500">Loading analytics...</div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* KPI Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Data Points</CardTitle>
            <Activity className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.totalDataPoints.toLocaleString()}</div>
            <p className="text-xs text-muted-foreground">
              <Badge variant="secondary" className="text-xs">
                {stats.validated} validated
              </Badge>
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Materials Tracked</CardTitle>
            <Building className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.materialsTracked}</div>
            <p className="text-xs text-muted-foreground">
              <TrendingUp className="inline h-3 w-3 mr-1" />
              Across all scopes
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Carbon Footprint</CardTitle>
            <TreePine className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.carbonFootprint.toLocaleString()}</div>
            <p className="text-xs text-muted-foreground">
              kg CO₂e total emissions
            </p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">ESG Coverage</CardTitle>
            <Award className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {categoryData.filter(c => c.value > 0).length}/6
            </div>
            <p className="text-xs text-muted-foreground">
              Categories with data
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Charts */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Monthly Data Entry Trends</CardTitle>
            <CardDescription>Data points collected over time</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <AreaChart data={monthlyData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="month" />
                <YAxis />
                <Tooltip />
                <Area type="monotone" dataKey="dataPoints" stroke="#3b82f6" fill="#3b82f6" fillOpacity={0.3} />
              </AreaChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Emissions by Scope</CardTitle>
            <CardDescription>GHG Protocol classification</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={scopeData}
                  cx="50%"
                  cy="50%"
                  labelLine={false}
                  label={({ name, value }) => `${name}: ${value}`}
                  outerRadius={80}
                  fill="#8884d8"
                  dataKey="value"
                >
                  {scopeData.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={entry.color} />
                  ))}
                </Pie>
                <Tooltip />
              </PieChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>ESG Category Distribution</CardTitle>
            <CardDescription>Data points by sustainability category</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={categoryData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="name" />
                <YAxis />
                <Tooltip />
                <Bar dataKey="value" fill="#10b981" />
              </BarChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Emissions Trend</CardTitle>
            <CardDescription>Monthly carbon footprint tracking</CardDescription>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <LineChart data={monthlyData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="month" />
                <YAxis />
                <Tooltip />
                <Line type="monotone" dataKey="emissions" stroke="#ef4444" strokeWidth={2} />
              </LineChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>
      </div>

      {/* Additional ESG Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Droplets className="h-5 w-5 text-blue-600" />
              Water Management
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {dataPoints.filter(d => d.metric_name?.toLowerCase().includes('water')).length}
            </div>
            <p className="text-sm text-muted-foreground">Water usage data points</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="h-5 w-5 text-green-600" />
              Social Impact
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {dataPoints.filter(d => 
                d.metric_name?.toLowerCase().includes('employee') || 
                d.metric_name?.toLowerCase().includes('diversity') ||
                d.metric_name?.toLowerCase().includes('training')
              ).length}
            </div>
            <p className="text-sm text-muted-foreground">Social metrics tracked</p>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Shield className="h-5 w-5 text-purple-600" />
              Governance
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {dataPoints.filter(d => 
                d.metric_name?.toLowerCase().includes('board') || 
                d.metric_name?.toLowerCase().includes('governance') ||
                d.metric_name?.toLowerCase().includes('ethics')
              ).length}
            </div>
            <p className="text-sm text-muted-foreground">Governance data points</p>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};
