
import React, { useState, useEffect } from 'react';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Building, Zap, Car, Target } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';

interface EmissionSummary {
  scope1: number;
  scope2: number;
  scope3: number;
  total: number;
  currency: string;
  totalLocalCurrency: number;
}

export const CarbonSummary: React.FC = () => {
  const [summary, setSummary] = useState<EmissionSummary>({
    scope1: 0,
    scope2: 0,
    scope3: 0,
    total: 0,
    currency: 'USD',
    totalLocalCurrency: 0
  });
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    fetchEmissionSummary();
  }, []);

  const fetchEmissionSummary = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data: calculations, error } = await supabase
        .from('carbon_calculations')
        .select('*')
        .eq('user_id', user.id);

      if (error) throw error;

      const scope1 = calculations?.filter(c => c.scope === 'scope-1').reduce((sum, c) => sum + Number(c.co2_equivalent), 0) || 0;
      const scope2 = calculations?.filter(c => c.scope === 'scope-2').reduce((sum, c) => sum + Number(c.co2_equivalent), 0) || 0;
      const scope3 = calculations?.filter(c => c.scope === 'scope-3').reduce((sum, c) => sum + Number(c.co2_equivalent), 0) || 0;
      const totalLocalCurrency = calculations?.reduce((sum, c) => sum + Number(c.local_amount || 0), 0) || 0;

      setSummary({
        scope1,
        scope2,
        scope3,
        total: scope1 + scope2 + scope3,
        currency: calculations?.[0]?.local_currency || 'USD',
        totalLocalCurrency
      });
    } catch (error) {
      console.error('Error fetching emission summary:', error);
    } finally {
      setIsLoading(false);
    }
  };

  if (isLoading) {
    return <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
      {[1, 2, 3, 4].map(i => (
        <Card key={i} className="p-4 animate-pulse">
          <div className="h-16 bg-gray-200 rounded"></div>
        </Card>
      ))}
    </div>;
  }

  return (
    <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
      <Card className="p-4 bg-green-50">
        <div className="flex items-center gap-2">
          <Building className="h-5 w-5 text-green-600" />
          <div>
            <p className="text-sm text-gray-600">Scope 1</p>
            <p className="text-xl font-bold">{summary.scope1.toFixed(2)}</p>
            <p className="text-sm">tCO2e</p>
          </div>
        </div>
        <Badge variant="outline" className="mt-2 text-xs">Direct Emissions</Badge>
      </Card>
      
      <Card className="p-4 bg-blue-50">
        <div className="flex items-center gap-2">
          <Zap className="h-5 w-5 text-blue-600" />
          <div>
            <p className="text-sm text-gray-600">Scope 2</p>
            <p className="text-xl font-bold">{summary.scope2.toFixed(2)}</p>
            <p className="text-sm">tCO2e</p>
          </div>
        </div>
        <Badge variant="outline" className="mt-2 text-xs">Energy Indirect</Badge>
      </Card>
      
      <Card className="p-4 bg-purple-50">
        <div className="flex items-center gap-2">
          <Car className="h-5 w-5 text-purple-600" />
          <div>
            <p className="text-sm text-gray-600">Scope 3</p>
            <p className="text-xl font-bold">{summary.scope3.toFixed(2)}</p>
            <p className="text-sm">tCO2e</p>
          </div>
        </div>
        <Badge variant="outline" className="mt-2 text-xs">Other Indirect</Badge>
      </Card>
      
      <Card className="p-4 bg-gray-50">
        <div className="flex items-center gap-2">
          <Target className="h-5 w-5 text-gray-600" />
          <div>
            <p className="text-sm text-gray-600">Total</p>
            <p className="text-xl font-bold">{summary.total.toFixed(2)}</p>
            <p className="text-sm">tCO2e</p>
          </div>
        </div>
        <Badge variant="outline" className="mt-2 text-xs">
          {summary.totalLocalCurrency.toFixed(0)} {summary.currency}
        </Badge>
      </Card>
    </div>
  );
};
