
import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Calculator, Flame } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface ChemicalCalculatorProps {
  onUpdate: () => void;
}

export const ChemicalCalculator: React.FC<ChemicalCalculatorProps> = ({ onUpdate }) => {
  const [activityName, setActivityName] = useState('');
  const [amount, setAmount] = useState('');
  const [chemicalType, setChemicalType] = useState('refrigerant');
  const [specificChemical, setSpecificChemical] = useState('R134a');
  const { toast } = useToast();

  const chemicalEmissionFactors = {
    refrigerant: {
      'R134a': { factor: 1430, unit: 'kgCO2e/kg', gwp: 1430 },
      'R22': { factor: 1810, unit: 'kgCO2e/kg', gwp: 1810 },
      'R410A': { factor: 2088, unit: 'kgCO2e/kg', gwp: 2088 },
      'R407C': { factor: 1774, unit: 'kgCO2e/kg', gwp: 1774 }
    },
    solvent: {
      'Acetone': { factor: 0.5, unit: 'kgCO2e/kg', gwp: 0.5 },
      'Methanol': { factor: 1.375, unit: 'kgCO2e/kg', gwp: 1.375 },
      'Ethanol': { factor: 1.913, unit: 'kgCO2e/kg', gwp: 1.913 },
      'Toluene': { factor: 2.934, unit: 'kgCO2e/kg', gwp: 2.934 }
    },
    cleaning_agent: {
      'Ammonia': { factor: 0, unit: 'kgCO2e/kg', gwp: 0 },
      'Bleach': { factor: 0.5, unit: 'kgCO2e/kg', gwp: 0.5 },
      'Detergent': { factor: 1.2, unit: 'kgCO2e/kg', gwp: 1.2 }
    }
  };

  const handleCalculate = async () => {
    if (!activityName || !amount) {
      toast({
        title: "Missing Information",
        description: "Please fill in all required fields",
        variant: "destructive"
      });
      return;
    }

    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('User not authenticated');

      const factor = chemicalEmissionFactors[chemicalType as keyof typeof chemicalEmissionFactors]?.[specificChemical as keyof typeof chemicalEmissionFactors.refrigerant];
      
      if (!factor) {
        toast({
          title: "Invalid Configuration",
          description: "Selected chemical configuration is not available",
          variant: "destructive"
        });
        return;
      }

      const co2Equivalent = parseFloat(amount) * factor.factor;

      const { error } = await supabase
        .from('carbon_calculations')
        .insert({
          user_id: user.id,
          calculation_type: 'chemical',
          scope: 'scope-1', // Most chemical emissions are direct
          activity_name: `${activityName} (${specificChemical})`,
          amount: parseFloat(amount),
          unit: factor.unit,
          emission_factor: factor.factor,
          co2_equivalent: co2Equivalent
        });

      if (error) throw error;

      toast({
        title: "Chemical Emission Calculated",
        description: `${co2Equivalent.toFixed(2)} kgCO2e from chemical usage`,
      });

      // Reset form
      setActivityName('');
      setAmount('');
      
      onUpdate();
    } catch (error) {
      console.error('Error calculating chemical emission:', error);
      toast({
        title: "Calculation Error",
        description: "Failed to calculate chemical emissions",
        variant: "destructive"
      });
    }
  };

  const getSpecificChemicals = () => {
    const chemicals = chemicalEmissionFactors[chemicalType as keyof typeof chemicalEmissionFactors];
    return Object.keys(chemicals || {});
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Flame className="h-5 w-5" />
          Chemical Emission Calculator
        </CardTitle>
        <p className="text-sm text-gray-600">
          Calculate emissions from chemical usage and leakage
        </p>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label>Activity Description</Label>
            <Input
              value={activityName}
              onChange={(e) => setActivityName(e.target.value)}
              placeholder="e.g., AC system refrigerant leak"
            />
          </div>
          <div>
            <Label>Amount (kg)</Label>
            <Input
              type="number"
              value={amount}
              onChange={(e) => setAmount(e.target.value)}
              placeholder="Enter quantity in kg"
            />
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label>Chemical Category</Label>
            <Select value={chemicalType} onValueChange={setChemicalType}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="refrigerant">Refrigerant</SelectItem>
                <SelectItem value="solvent">Solvent</SelectItem>
                <SelectItem value="cleaning_agent">Cleaning Agent</SelectItem>
              </SelectContent>
            </Select>
          </div>
          <div>
            <Label>Specific Chemical</Label>
            <Select value={specificChemical} onValueChange={setSpecificChemical}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {getSpecificChemicals().map(chemical => (
                  <SelectItem key={chemical} value={chemical}>
                    {chemical}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        <div className="p-4 bg-orange-50 rounded-lg">
          <p className="text-sm">
            <strong>Global Warming Potential (GWP):</strong> {
              chemicalEmissionFactors[chemicalType as keyof typeof chemicalEmissionFactors]?.[specificChemical as keyof typeof chemicalEmissionFactors.refrigerant]?.gwp || 'N/A'
            }
          </p>
          <p className="text-sm mt-1">
            <strong>Emission Factor:</strong> {
              chemicalEmissionFactors[chemicalType as keyof typeof chemicalEmissionFactors]?.[specificChemical as keyof typeof chemicalEmissionFactors.refrigerant]?.factor || 'N/A'
            } {
              chemicalEmissionFactors[chemicalType as keyof typeof chemicalEmissionFactors]?.[specificChemical as keyof typeof chemicalEmissionFactors.refrigerant]?.unit || ''
            }
          </p>
        </div>

        <Button onClick={handleCalculate} className="w-full">
          <Calculator className="h-4 w-4 mr-2" />
          Calculate Chemical Emissions
        </Button>
      </CardContent>
    </Card>
  );
};
