
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { Calculator, Zap, Building, Car } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface EnergyCalculatorProps {
  onUpdate: () => void;
}

interface CurrencyRate {
  currency_code: string;
  usd_rate: number;
  country: string;
}

export const EnergyCalculator: React.FC<EnergyCalculatorProps> = ({ onUpdate }) => {
  const [currencyRates, setCurrencyRates] = useState<CurrencyRate[]>([]);
  const [selectedCurrency, setSelectedCurrency] = useState('USD');
  const [calculations, setCalculations] = useState<any[]>([]);
  const { toast } = useToast();

  // Emission factors for energy sources
  const energyEmissionFactors = {
    // Scope 1 - Direct emissions
    naturalGas: { factor: 0.202, unit: 'kgCO2e/kWh', scope: 'scope-1' },
    diesel: { factor: 2.68, unit: 'kgCO2e/liter', scope: 'scope-1' },
    gasoline: { factor: 2.31, unit: 'kgCO2e/liter', scope: 'scope-1' },
    propane: { factor: 1.51, unit: 'kgCO2e/liter', scope: 'scope-1' },
    coal: { factor: 0.354, unit: 'kgCO2e/kWh', scope: 'scope-1' },
    
    // Scope 2 - Indirect emissions from purchased energy
    electricity: { factor: 0.855, unit: 'lbCO2e/kWh', scope: 'scope-2' },
    steam: { factor: 0.2, unit: 'kgCO2e/kWh', scope: 'scope-2' },
    heating: { factor: 0.25, unit: 'kgCO2e/kWh', scope: 'scope-2' },
    cooling: { factor: 0.18, unit: 'kgCO2e/kWh', scope: 'scope-2' }
  };

  useEffect(() => {
    fetchCurrencyRates();
    fetchCalculations();
  }, []);

  const fetchCurrencyRates = async () => {
    try {
      const { data, error } = await supabase
        .from('currency_rates')
        .select('*')
        .order('currency_code');

      if (error) throw error;
      setCurrencyRates(data || []);
    } catch (error) {
      console.error('Error fetching currency rates:', error);
    }
  };

  const fetchCalculations = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from('carbon_calculations')
        .select('*')
        .eq('user_id', user.id)
        .eq('calculation_type', 'energy')
        .order('created_at', { ascending: false });

      if (error) throw error;
      setCalculations(data || []);
    } catch (error) {
      console.error('Error fetching calculations:', error);
    }
  };

  const calculateEmission = async (
    activityName: string,
    amount: number,
    energyType: keyof typeof energyEmissionFactors,
    localAmount?: number
  ) => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('User not authenticated');

      const factor = energyEmissionFactors[energyType];
      const co2Equivalent = amount * factor.factor;
      
      // Convert to USD if needed
      const exchangeRate = currencyRates.find(r => r.currency_code === selectedCurrency)?.usd_rate || 1;
      const usdAmount = localAmount ? localAmount * exchangeRate : undefined;

      const { error } = await supabase
        .from('carbon_calculations')
        .insert({
          user_id: user.id,
          calculation_type: 'energy',
          scope: factor.scope,
          activity_name: activityName,
          amount,
          unit: factor.unit,
          emission_factor: factor.factor,
          co2_equivalent: co2Equivalent,
          local_currency: selectedCurrency,
          local_amount: localAmount,
          usd_amount: usdAmount,
          exchange_rate: exchangeRate
        });

      if (error) throw error;

      toast({
        title: "Calculation Added",
        description: `${activityName}: ${co2Equivalent.toFixed(2)} kgCO2e`,
      });

      fetchCalculations();
      onUpdate();
    } catch (error) {
      console.error('Error calculating emission:', error);
      toast({
        title: "Error",
        description: "Failed to calculate emission",
        variant: "destructive"
      });
    }
  };

  const Scope1Calculator = () => {
    const [activityName, setActivityName] = useState('');
    const [amount, setAmount] = useState('');
    const [localAmount, setLocalAmount] = useState('');
    const [fuelType, setFuelType] = useState<keyof typeof energyEmissionFactors>('naturalGas');

    const handleCalculate = () => {
      if (activityName && amount) {
        calculateEmission(
          activityName,
          parseFloat(amount),
          fuelType,
          localAmount ? parseFloat(localAmount) : undefined
        );
        setActivityName('');
        setAmount('');
        setLocalAmount('');
      }
    };

    const scope1Fuels = Object.keys(energyEmissionFactors).filter(
      key => energyEmissionFactors[key as keyof typeof energyEmissionFactors].scope === 'scope-1'
    );

    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg flex items-center gap-2">
            <Building className="h-5 w-5" />
            Scope 1: Direct Energy Emissions
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <Label>Activity Description</Label>
              <Input
                value={activityName}
                onChange={(e) => setActivityName(e.target.value)}
                placeholder="e.g., Office heating with natural gas"
              />
            </div>
            <div>
              <Label>Fuel Type</Label>
              <Select value={fuelType} onValueChange={(value: keyof typeof energyEmissionFactors) => setFuelType(value)}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {scope1Fuels.map(fuel => (
                    <SelectItem key={fuel} value={fuel}>
                      {fuel.charAt(0).toUpperCase() + fuel.slice(1).replace(/([A-Z])/g, ' $1')}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
              <Label>Amount</Label>
              <Input
                type="number"
                value={amount}
                onChange={(e) => setAmount(e.target.value)}
                placeholder="Enter quantity"
              />
              <p className="text-xs text-gray-500 mt-1">
                Unit: {energyEmissionFactors[fuelType].unit.split('/')[1]}
              </p>
            </div>
            <div>
              <Label>Cost in Local Currency (Optional)</Label>
              <Input
                type="number"
                value={localAmount}
                onChange={(e) => setLocalAmount(e.target.value)}
                placeholder="Enter cost"
              />
            </div>
            <div>
              <Label>Currency</Label>
              <Select value={selectedCurrency} onValueChange={setSelectedCurrency}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {currencyRates.map(rate => (
                    <SelectItem key={rate.currency_code} value={rate.currency_code}>
                      {rate.currency_code} - {rate.country}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>

          <Button onClick={handleCalculate} className="w-full">
            <Calculator className="h-4 w-4 mr-2" />
            Calculate Scope 1 Emissions
          </Button>
        </CardContent>
      </Card>
    );
  };

  const Scope2Calculator = () => {
    const [activityName, setActivityName] = useState('');
    const [amount, setAmount] = useState('');
    const [localAmount, setLocalAmount] = useState('');
    const [energyType, setEnergyType] = useState<keyof typeof energyEmissionFactors>('electricity');

    const handleCalculate = () => {
      if (activityName && amount) {
        calculateEmission(
          activityName,
          parseFloat(amount),
          energyType,
          localAmount ? parseFloat(localAmount) : undefined
        );
        setActivityName('');
        setAmount('');
        setLocalAmount('');
      }
    };

    const scope2Energy = Object.keys(energyEmissionFactors).filter(
      key => energyEmissionFactors[key as keyof typeof energyEmissionFactors].scope === 'scope-2'
    );

    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-lg flex items-center gap-2">
            <Zap className="h-5 w-5" />
            Scope 2: Indirect Energy Emissions
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <Label>Activity Description</Label>
              <Input
                value={activityName}
                onChange={(e) => setActivityName(e.target.value)}
                placeholder="e.g., Office electricity consumption"
              />
            </div>
            <div>
              <Label>Energy Type</Label>
              <Select value={energyType} onValueChange={(value: keyof typeof energyEmissionFactors) => setEnergyType(value)}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {scope2Energy.map(energy => (
                    <SelectItem key={energy} value={energy}>
                      {energy.charAt(0).toUpperCase() + energy.slice(1)}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
              <Label>Amount (kWh)</Label>
              <Input
                type="number"
                value={amount}
                onChange={(e) => setAmount(e.target.value)}
                placeholder="Enter kWh consumed"
              />
            </div>
            <div>
              <Label>Cost in Local Currency (Optional)</Label>
              <Input
                type="number"
                value={localAmount}
                onChange={(e) => setLocalAmount(e.target.value)}
                placeholder="Enter cost"
              />
            </div>
            <div>
              <Label>Currency</Label>
              <Select value={selectedCurrency} onValueChange={setSelectedCurrency}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {currencyRates.map(rate => (
                    <SelectItem key={rate.currency_code} value={rate.currency_code}>
                      {rate.currency_code} - {rate.country}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>

          <Button onClick={handleCalculate} className="w-full">
            <Calculator className="h-4 w-4 mr-2" />
            Calculate Scope 2 Emissions
          </Button>
        </CardContent>
      </Card>
    );
  };

  const CalculationHistory = () => {
    const scope1Calcs = calculations.filter(c => c.scope === 'scope-1');
    const scope2Calcs = calculations.filter(c => c.scope === 'scope-2');

    return (
      <div className="space-y-4">
        <h4 className="font-semibold">Recent Energy Calculations</h4>
        
        {scope1Calcs.length > 0 && (
          <div>
            <h5 className="font-medium text-green-700 mb-2">Scope 1 Calculations</h5>
            {scope1Calcs.slice(0, 5).map((calc) => (
              <Card key={calc.id} className="p-3 mb-2">
                <div className="flex justify-between items-center">
                  <div>
                    <p className="font-medium">{calc.activity_name}</p>
                    <p className="text-sm text-gray-600">
                      {calc.amount} × {calc.emission_factor} = {Number(calc.co2_equivalent).toFixed(2)} kgCO2e
                    </p>
                    {calc.local_amount && (
                      <p className="text-xs text-gray-500">
                        Cost: {Number(calc.local_amount).toFixed(2)} {calc.local_currency}
                      </p>
                    )}
                  </div>
                  <Badge variant="outline" className="bg-green-50">
                    {new Date(calc.created_at).toLocaleDateString()}
                  </Badge>
                </div>
              </Card>
            ))}
          </div>
        )}

        {scope2Calcs.length > 0 && (
          <div>
            <h5 className="font-medium text-blue-700 mb-2">Scope 2 Calculations</h5>
            {scope2Calcs.slice(0, 5).map((calc) => (
              <Card key={calc.id} className="p-3 mb-2">
                <div className="flex justify-between items-center">
                  <div>
                    <p className="font-medium">{calc.activity_name}</p>
                    <p className="text-sm text-gray-600">
                      {calc.amount} × {calc.emission_factor} = {Number(calc.co2_equivalent).toFixed(2)} kgCO2e
                    </p>
                    {calc.local_amount && (
                      <p className="text-xs text-gray-500">
                        Cost: {Number(calc.local_amount).toFixed(2)} {calc.local_currency}
                      </p>
                    )}
                  </div>
                  <Badge variant="outline" className="bg-blue-50">
                    {new Date(calc.created_at).toLocaleDateString()}
                  </Badge>
                </div>
              </Card>
            ))}
          </div>
        )}
      </div>
    );
  };

  return (
    <div className="space-y-6">
      <Tabs defaultValue="scope1">
        <TabsList className="grid w-full grid-cols-3">
          <TabsTrigger value="scope1">Scope 1 - Direct</TabsTrigger>
          <TabsTrigger value="scope2">Scope 2 - Indirect Energy</TabsTrigger>
          <TabsTrigger value="history">Calculation History</TabsTrigger>
        </TabsList>
        
        <TabsContent value="scope1">
          <Scope1Calculator />
        </TabsContent>
        
        <TabsContent value="scope2">
          <Scope2Calculator />
        </TabsContent>
        
        <TabsContent value="history">
          <CalculationHistory />
        </TabsContent>
      </Tabs>
    </div>
  );
};
