
import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Calculator, Factory } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface ProductionCalculatorProps {
  onUpdate: () => void;
}

export const ProductionCalculator: React.FC<ProductionCalculatorProps> = ({ onUpdate }) => {
  const [processName, setProcessName] = useState('');
  const [productionAmount, setProductionAmount] = useState('');
  const [processType, setProcessType] = useState('cement');
  const [energyInput, setEnergyInput] = useState('');
  const [rawMaterialInput, setRawMaterialInput] = useState('');
  const { toast } = useToast();

  const productionEmissionFactors = {
    cement: {
      process: { factor: 0.54, unit: 'tCO2e/tonne cement' },
      energy: { factor: 0.1, unit: 'tCO2e/GJ' }
    },
    steel: {
      process: { factor: 1.85, unit: 'tCO2e/tonne steel' },
      energy: { factor: 0.08, unit: 'tCO2e/GJ' }
    },
    aluminum: {
      process: { factor: 1.69, unit: 'tCO2e/tonne aluminum' },
      energy: { factor: 0.12, unit: 'tCO2e/GJ' }
    },
    paper: {
      process: { factor: 0.7, unit: 'tCO2e/tonne paper' },
      energy: { factor: 0.05, unit: 'tCO2e/GJ' }
    },
    chemical: {
      process: { factor: 2.1, unit: 'tCO2e/tonne product' },
      energy: { factor: 0.15, unit: 'tCO2e/GJ' }
    }
  };

  const handleCalculate = async () => {
    if (!processName || !productionAmount) {
      toast({
        title: "Missing Information",
        description: "Please fill in process name and production amount",
        variant: "destructive"
      });
      return;
    }

    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('User not authenticated');

      const factors = productionEmissionFactors[processType as keyof typeof productionEmissionFactors];
      
      // Calculate process emissions
      const processEmissions = parseFloat(productionAmount) * factors.process.factor;
      
      // Calculate energy emissions if provided
      let energyEmissions = 0;
      if (energyInput) {
        energyEmissions = parseFloat(energyInput) * factors.energy.factor;
      }

      const totalEmissions = processEmissions + energyEmissions;

      const { error } = await supabase
        .from('carbon_calculations')
        .insert({
          user_id: user.id,
          calculation_type: 'production',
          scope: 'scope-1', // Production processes are typically direct emissions
          activity_name: `${processName} (${processType} production)`,
          amount: parseFloat(productionAmount),
          unit: factors.process.unit,
          emission_factor: factors.process.factor,
          co2_equivalent: totalEmissions
        });

      if (error) throw error;

      toast({
        title: "Production Emission Calculated",
        description: `${totalEmissions.toFixed(2)} tCO2e from production process`,
      });

      // Reset form
      setProcessName('');
      setProductionAmount('');
      setEnergyInput('');
      setRawMaterialInput('');
      
      onUpdate();
    } catch (error) {
      console.error('Error calculating production emission:', error);
      toast({
        title: "Calculation Error",
        description: "Failed to calculate production emissions",
        variant: "destructive"
      });
    }
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Factory className="h-5 w-5" />
          Production Process Calculator
        </CardTitle>
        <p className="text-sm text-gray-600">
          Calculate emissions from industrial production processes
        </p>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label>Process Name</Label>
            <Input
              value={processName}
              onChange={(e) => setProcessName(e.target.value)}
              placeholder="e.g., Cement manufacturing line 1"
            />
          </div>
          <div>
            <Label>Process Type</Label>
            <Select value={processType} onValueChange={setProcessType}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="cement">Cement Production</SelectItem>
                <SelectItem value="steel">Steel Production</SelectItem>
                <SelectItem value="aluminum">Aluminum Production</SelectItem>
                <SelectItem value="paper">Paper Production</SelectItem>
                <SelectItem value="chemical">Chemical Production</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label>Production Amount (tonnes)</Label>
            <Input
              type="number"
              value={productionAmount}
              onChange={(e) => setProductionAmount(e.target.value)}
              placeholder="Enter production quantity"
            />
          </div>
          <div>
            <Label>Energy Input (GJ) - Optional</Label>
            <Input
              type="number"
              value={energyInput}
              onChange={(e) => setEnergyInput(e.target.value)}
              placeholder="Enter energy consumption"
            />
          </div>
        </div>

        <div>
          <Label>Raw Material Notes (Optional)</Label>
          <Input
            value={rawMaterialInput}
            onChange={(e) => setRawMaterialInput(e.target.value)}
            placeholder="e.g., Limestone, iron ore quantities"
          />
        </div>

        <div className="p-4 bg-blue-50 rounded-lg">
          <p className="text-sm">
            <strong>Process Emission Factor:</strong> {
              productionEmissionFactors[processType as keyof typeof productionEmissionFactors]?.process.factor
            } {
              productionEmissionFactors[processType as keyof typeof productionEmissionFactors]?.process.unit
            }
          </p>
          <p className="text-sm mt-1">
            <strong>Energy Emission Factor:</strong> {
              productionEmissionFactors[processType as keyof typeof productionEmissionFactors]?.energy.factor
            } {
              productionEmissionFactors[processType as keyof typeof productionEmissionFactors]?.energy.unit
            }
          </p>
        </div>

        <Button onClick={handleCalculate} className="w-full">
          <Calculator className="h-4 w-4 mr-2" />
          Calculate Production Emissions
        </Button>
      </CardContent>
    </Card>
  );
};
