
import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Calculator, Recycle } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface WasteCalculatorProps {
  onUpdate: () => void;
}

export const WasteCalculator: React.FC<WasteCalculatorProps> = ({ onUpdate }) => {
  const [activityName, setActivityName] = useState('');
  const [amount, setAmount] = useState('');
  const [wasteType, setWasteType] = useState('general');
  const [disposalMethod, setDisposalMethod] = useState('landfill');
  const { toast } = useToast();

  const wasteEmissionFactors = {
    general: {
      landfill: { factor: 0.57, unit: 'tCO2e/tonne' },
      incineration: { factor: 0.21, unit: 'tCO2e/tonne' },
      recycling: { factor: 0.05, unit: 'tCO2e/tonne' }
    },
    plastic: {
      landfill: { factor: 1.8, unit: 'tCO2e/tonne' },
      incineration: { factor: 2.1, unit: 'tCO2e/tonne' },
      recycling: { factor: 0.1, unit: 'tCO2e/tonne' }
    },
    organic: {
      landfill: { factor: 1.2, unit: 'tCO2e/tonne' },
      composting: { factor: 0.3, unit: 'tCO2e/tonne' },
      anaerobic_digestion: { factor: 0.1, unit: 'tCO2e/tonne' }
    },
    paper: {
      landfill: { factor: 0.8, unit: 'tCO2e/tonne' },
      incineration: { factor: 0.4, unit: 'tCO2e/tonne' },
      recycling: { factor: 0.02, unit: 'tCO2e/tonne' }
    }
  };

  const handleCalculate = async () => {
    if (!activityName || !amount) {
      toast({
        title: "Missing Information",
        description: "Please fill in all required fields",
        variant: "destructive"
      });
      return;
    }

    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('User not authenticated');

      const factor = wasteEmissionFactors[wasteType as keyof typeof wasteEmissionFactors]?.[disposalMethod as keyof typeof wasteEmissionFactors.general];
      
      if (!factor) {
        toast({
          title: "Invalid Configuration",
          description: "Selected waste type and disposal method combination is not available",
          variant: "destructive"
        });
        return;
      }

      const co2Equivalent = parseFloat(amount) * factor.factor;

      const { error } = await supabase
        .from('carbon_calculations')
        .insert({
          user_id: user.id,
          calculation_type: 'waste',
          scope: 'scope-3',
          activity_name: `${activityName} (${wasteType} - ${disposalMethod})`,
          amount: parseFloat(amount),
          unit: factor.unit,
          emission_factor: factor.factor,
          co2_equivalent: co2Equivalent
        });

      if (error) throw error;

      toast({
        title: "Waste Emission Calculated",
        description: `${co2Equivalent.toFixed(2)} tCO2e from waste disposal`,
      });

      // Reset form
      setActivityName('');
      setAmount('');
      setWasteType('general');
      setDisposalMethod('landfill');
      
      onUpdate();
    } catch (error) {
      console.error('Error calculating waste emission:', error);
      toast({
        title: "Calculation Error",
        description: "Failed to calculate waste emissions",
        variant: "destructive"
      });
    }
  };

  const getDisposalMethods = () => {
    const methods = wasteEmissionFactors[wasteType as keyof typeof wasteEmissionFactors];
    return Object.keys(methods || {});
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Recycle className="h-5 w-5" />
          Waste Emission Calculator
        </CardTitle>
        <p className="text-sm text-gray-600">
          Calculate emissions from waste generation and disposal (Scope 3)
        </p>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label>Activity Description</Label>
            <Input
              value={activityName}
              onChange={(e) => setActivityName(e.target.value)}
              placeholder="e.g., Office waste disposal"
            />
          </div>
          <div>
            <Label>Amount (tonnes)</Label>
            <Input
              type="number"
              value={amount}
              onChange={(e) => setAmount(e.target.value)}
              placeholder="Enter waste quantity in tonnes"
            />
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label>Waste Type</Label>
            <Select value={wasteType} onValueChange={setWasteType}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="general">General Waste</SelectItem>
                <SelectItem value="plastic">Plastic Waste</SelectItem>
                <SelectItem value="organic">Organic Waste</SelectItem>
                <SelectItem value="paper">Paper Waste</SelectItem>
              </SelectContent>
            </Select>
          </div>
          <div>
            <Label>Disposal Method</Label>
            <Select value={disposalMethod} onValueChange={setDisposalMethod}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {getDisposalMethods().map(method => (
                  <SelectItem key={method} value={method}>
                    {method.charAt(0).toUpperCase() + method.slice(1).replace(/_/g, ' ')}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        <div className="p-4 bg-gray-50 rounded-lg">
          <p className="text-sm">
            <strong>Emission Factor:</strong> {
              wasteEmissionFactors[wasteType as keyof typeof wasteEmissionFactors]?.[disposalMethod as keyof typeof wasteEmissionFactors.general]?.factor || 'N/A'
            } {
              wasteEmissionFactors[wasteType as keyof typeof wasteEmissionFactors]?.[disposalMethod as keyof typeof wasteEmissionFactors.general]?.unit || ''
            }
          </p>
        </div>

        <Button onClick={handleCalculate} className="w-full">
          <Calculator className="h-4 w-4 mr-2" />
          Calculate Waste Emissions
        </Button>
      </CardContent>
    </Card>
  );
};
