
import React from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { TrendingUp, TrendingDown, AlertCircle, CheckCircle, Target, Leaf } from 'lucide-react';

export const DashboardOverview = () => {
  const kpis = [
    {
      title: 'Total Carbon Footprint',
      value: '1,247 tCO2e',
      change: '-12%',
      trend: 'down',
      status: 'good'
    },
    {
      title: 'ESG Score',
      value: '78/100',
      change: '+5pts',
      trend: 'up',
      status: 'good'
    },
    {
      title: 'Net-Zero Progress',
      value: '43%',
      change: '+8%',
      trend: 'up',
      status: 'on-track'
    },
    {
      title: 'Risk Level',
      value: 'Medium',
      change: 'Stable',
      trend: 'stable',
      status: 'warning'
    }
  ];

  const recentReports = [
    { name: 'IFRS S1 Sustainability Report', status: 'Complete', date: '2024-01-15' },
    { name: 'GRI Standards Report', status: 'In Progress', date: '2024-01-20' },
    { name: 'TCFD Climate Report', status: 'Pending', date: '2024-01-25' }
  ];

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">ESG Dashboard</h1>
          <p className="text-gray-600 mt-1">Monitor your sustainability performance across all ESG metrics</p>
        </div>
        <Badge variant="outline" className="bg-green-50 text-green-700 border-green-200">
          Last Updated: Today 14:30
        </Badge>
      </div>

      {/* KPI Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {kpis.map((kpi) => (
          <Card key={kpi.title} className="relative overflow-hidden">
            <CardHeader className="pb-2">
              <CardTitle className="text-sm font-medium text-gray-600">{kpi.title}</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="flex items-center justify-between">
                <div>
                  <div className="text-2xl font-bold text-gray-900">{kpi.value}</div>
                  <div className={`flex items-center gap-1 text-sm ${
                    kpi.status === 'good' ? 'text-green-600' : 
                    kpi.status === 'warning' ? 'text-yellow-600' : 'text-red-600'
                  }`}>
                    {kpi.trend === 'up' && <TrendingUp className="h-4 w-4" />}
                    {kpi.trend === 'down' && <TrendingDown className="h-4 w-4" />}
                    <span>{kpi.change}</span>
                  </div>
                </div>
                <div className={`p-2 rounded-full ${
                  kpi.status === 'good' ? 'bg-green-100' : 
                  kpi.status === 'warning' ? 'bg-yellow-100' : 'bg-red-100'
                }`}>
                  {kpi.status === 'good' && <CheckCircle className="h-6 w-6 text-green-600" />}
                  {kpi.status === 'warning' && <AlertCircle className="h-6 w-6 text-yellow-600" />}
                  {kpi.status === 'on-track' && <Target className="h-6 w-6 text-blue-600" />}
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Main Content Tabs */}
      <Tabs defaultValue="overview" className="space-y-4">
        <TabsList>
          <TabsTrigger value="overview">Overview</TabsTrigger>
          <TabsTrigger value="emissions">Emissions</TabsTrigger>
          <TabsTrigger value="reports">Reports</TabsTrigger>
          <TabsTrigger value="targets">Targets</TabsTrigger>
        </TabsList>

        <TabsContent value="overview" className="space-y-4">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Emissions Breakdown */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Leaf className="h-5 w-5 text-green-600" />
                  Emissions by Scope
                </CardTitle>
                <CardDescription>GHG Protocol categorization</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div>
                    <div className="flex justify-between text-sm mb-1">
                      <span>Scope 1 (Direct)</span>
                      <span>342 tCO2e</span>
                    </div>
                    <Progress value={27} className="h-2" />
                  </div>
                  <div>
                    <div className="flex justify-between text-sm mb-1">
                      <span>Scope 2 (Electricity)</span>
                      <span>456 tCO2e</span>
                    </div>
                    <Progress value={37} className="h-2" />
                  </div>
                  <div>
                    <div className="flex justify-between text-sm mb-1">
                      <span>Scope 3 (Value Chain)</span>
                      <span>449 tCO2e</span>
                    </div>
                    <Progress value={36} className="h-2" />
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Recent Reports */}
            <Card>
              <CardHeader>
                <CardTitle>Recent Reports</CardTitle>
                <CardDescription>Latest sustainability reporting activity</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {recentReports.map((report, index) => (
                    <div key={index} className="flex items-center justify-between p-3 border rounded-lg">
                      <div>
                        <div className="font-medium text-sm">{report.name}</div>
                        <div className="text-xs text-gray-500">{report.date}</div>
                      </div>
                      <Badge 
                        variant={
                          report.status === 'Complete' ? 'default' :
                          report.status === 'In Progress' ? 'secondary' : 'outline'
                        }
                        className={
                          report.status === 'Complete' ? 'bg-green-100 text-green-800' :
                          report.status === 'In Progress' ? 'bg-blue-100 text-blue-800' : ''
                        }
                      >
                        {report.status}
                      </Badge>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="emissions">
          <Card>
            <CardHeader>
              <CardTitle>Detailed Emissions Analysis</CardTitle>
              <CardDescription>
                Comprehensive breakdown of your carbon footprint across all operational areas
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-64 flex items-center justify-center text-gray-500">
                Interactive emissions charts will be displayed here
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="reports">
          <Card>
            <CardHeader>
              <CardTitle>Report Management</CardTitle>
              <CardDescription>
                Generate and manage compliance reports across multiple standards
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-64 flex items-center justify-center text-gray-500">
                Report generation interface will be displayed here
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="targets">
          <Card>
            <CardHeader>
              <CardTitle>Net-Zero Targets</CardTitle>
              <CardDescription>
                Track progress against your sustainability goals and SBTi commitments
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="h-64 flex items-center justify-center text-gray-500">
                Target tracking dashboard will be displayed here
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
};
