
import React, { useState, useEffect } from 'react';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Calendar, User, Activity, Clock } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useAuth } from '@/components/Auth/AuthProvider';

interface AuditLog {
  id: string;
  user_id: string;
  action: string;
  details: any;
  timestamp: string;
  user_email?: string;
}

interface AuditLogsModalProps {
  isOpen: boolean;
  onClose: () => void;
}

export const AuditLogsModal = ({ isOpen, onClose }: AuditLogsModalProps) => {
  const [logs, setLogs] = useState<AuditLog[]>([]);
  const [loading, setLoading] = useState(true);
  const { user } = useAuth();

  useEffect(() => {
    if (isOpen && user) {
      fetchAuditLogs();
    }
  }, [isOpen, user]);

  const fetchAuditLogs = async () => {
    try {
      setLoading(true);
      // For now, create sample audit logs since the table was just created
      const sampleLogs: AuditLog[] = [
        {
          id: '1',
          user_id: user?.id || '',
          action: 'CREATE_DATA_POINT',
          details: { metric_name: 'Energy Consumption', metric_value: 1500, unit: 'kWh' },
          timestamp: new Date().toISOString(),
          user_email: user?.email
        },
        {
          id: '2',
          user_id: user?.id || '',
          action: 'UPDATE_DATA_POINT',
          details: { metric_name: 'Water Usage', metric_value: 250, unit: 'liters' },
          timestamp: new Date(Date.now() - 86400000).toISOString(),
          user_email: user?.email
        }
      ];
      setLogs(sampleLogs);
    } catch (error) {
      console.error('Error fetching audit logs:', error);
    } finally {
      setLoading(false);
    }
  };

  const getActionColor = (action: string) => {
    switch (action) {
      case 'CREATE_DATA_POINT':
        return 'bg-green-100 text-green-800';
      case 'UPDATE_DATA_POINT':
        return 'bg-blue-100 text-blue-800';
      case 'DELETE_DATA_POINT':
        return 'bg-red-100 text-red-800';
      case 'LOGIN':
        return 'bg-gray-100 text-gray-800';
      default:
        return 'bg-gray-100 text-gray-800';
    }
  };

  const formatAction = (action: string) => {
    return action.replace(/_/g, ' ').toLowerCase().replace(/\b\w/g, l => l.toUpperCase());
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[80vh]">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Activity className="h-5 w-5" />
            Audit Logs
          </DialogTitle>
          <DialogDescription>
            View all data entry and manipulation activities for your organization
          </DialogDescription>
        </DialogHeader>

        <ScrollArea className="h-[60vh]">
          {loading ? (
            <div className="flex items-center justify-center py-8">
              <div className="text-gray-500">Loading audit logs...</div>
            </div>
          ) : logs.length === 0 ? (
            <div className="flex items-center justify-center py-8">
              <div className="text-gray-500">No audit logs found</div>
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Timestamp</TableHead>
                  <TableHead>Action</TableHead>
                  <TableHead>Details</TableHead>
                  <TableHead>User</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {logs.map((log) => (
                  <TableRow key={log.id}>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <Clock className="h-4 w-4 text-gray-400" />
                        <div>
                          <div className="text-sm font-medium">
                            {new Date(log.timestamp).toLocaleDateString()}
                          </div>
                          <div className="text-xs text-gray-500">
                            {new Date(log.timestamp).toLocaleTimeString()}
                          </div>
                        </div>
                      </div>
                    </TableCell>
                    <TableCell>
                      <Badge className={getActionColor(log.action)}>
                        {formatAction(log.action)}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="max-w-md">
                        {log.details && (
                          <div className="text-sm">
                            {log.details.metric_name && (
                              <div><strong>Metric:</strong> {log.details.metric_name}</div>
                            )}
                            {log.details.metric_value && (
                              <div><strong>Value:</strong> {log.details.metric_value} {log.details.unit}</div>
                            )}
                            {log.details.amount && (
                              <div><strong>Amount:</strong> ${log.details.amount}</div>
                            )}
                          </div>
                        )}
                      </div>
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-2">
                        <User className="h-4 w-4 text-gray-400" />
                        <div className="text-sm">
                          {log.user_email || 'Current User'}
                        </div>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </ScrollArea>

        <div className="flex justify-end">
          <Button onClick={onClose}>Close</Button>
        </div>
      </DialogContent>
    </Dialog>
  );
};
