
import React, { useState } from 'react';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Upload, FileText, AlertCircle, Flame, Zap, Camera,  Scan, Receipt } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useAuth } from '@/components/Auth/AuthProvider';
import { materialMetrics } from '@/data/materialMetrics';

interface DataEntryModalProps {
  isOpen: boolean;
  onClose: () => void;
  onSuccess?: () => void;
}

export const DataEntryModal = ({ isOpen, onClose, onSuccess }: DataEntryModalProps) => {
  const [activeTab, setActiveTab] = useState('manual');
  const [file, setFile] = useState<File | null>(null);
  const [proofFile, setProofFile] = useState<File | null>(null);
  const [metricName, setMetricName] = useState('');
  const [metricValue, setMetricValue] = useState('');
  const [unit, setUnit] = useState('');
  const [amount, setAmount] = useState('');
  const [quantity, setQuantity] = useState('');
  const [scope, setScope] = useState('');
  const [category, setCategory] = useState('');
  const [reportingPeriod, setReportingPeriod] = useState('');
  const [dataSource, setDataSource] = useState('');
  const [selectedMaterial, setSelectedMaterial] = useState('');
  const [isCustomEntry, setIsCustomEntry] = useState(true);
  const [ocrText, setOcrText] = useState('');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const { user } = useAuth();

  const unitOptions = {
    energy: ['kWh', 'MWh', 'GJ', 'BTU', 'therms'],
    volume: ['liters', 'gallons', 'm³', 'barrels'],
    weight: ['kg', 'tonnes', 'lbs', 'tons'],
    area: ['m²', 'hectares', 'acres', 'ft²'],
    time: ['hours', 'days', 'months', 'years'],
    currency: ['USD', 'EUR', 'GBP', 'CAD'],
    count: ['units', 'pieces', 'employees', 'incidents']
  };

  const handleFileUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0];
    if (selectedFile) {
      setFile(selectedFile);
      setDataSource('File Upload');
    }
  };

  const handleProofUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0];
    if (selectedFile) {
      setProofFile(selectedFile);
    }
  };

  const handleMaterialSelect = (value: string) => {
    if (value === 'custom') {
      setIsCustomEntry(true);
      setSelectedMaterial('');
      setMetricName('');
      setUnit('');
      setScope('');
      setCategory('');
    } else {
      const material = materialMetrics.find(m => m.id === value);
      if (material) {
        setIsCustomEntry(false);
        setSelectedMaterial(value);
        setMetricName(material.name);
        setUnit(material.unit);
        setScope(material.scope);
        setCategory(material.category);
      }
    }
  };

  const handleManualSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!user) return;

    // Validation
    if (!metricName || !metricValue || !unit) {
      setError('Please fill in all required fields (Metric Name, Value, and Unit)');
      return;
    }

    setLoading(true);
    setError('');

    try {
      const selectedMaterialData = materialMetrics.find(m => m.id === selectedMaterial);
      const calculatedEmissions = selectedMaterialData 
        ? parseFloat(metricValue) * selectedMaterialData.emissionFactor
        : null;

      const { error: dataError } = await supabase
        .from('esg_data_points')
        .insert([
          {
            user_id: user.id,
            metric_name: metricName,
            metric_value: parseFloat(metricValue),
            unit,
            scope,
            reporting_period: reportingPeriod,
            data_source: dataSource || 'Manual Entry',
            validation_status: 'pending'
          }
        ]);

      if (dataError) throw dataError;

      // Log the action for audit purposes (simplified for now)
      console.log('Data entry logged:', {
        action: 'CREATE_DATA_POINT',
        user_id: user.id,
        details: { metric_name: metricName, metric_value: metricValue, unit }
      });

      // Reset form
      setMetricName('');
      setMetricValue('');
      setUnit('');
      setAmount('');
      setQuantity('');
      setScope('');
      setReportingPeriod('');
      setDataSource('');
      setSelectedMaterial('');
      setIsCustomEntry(true);
      setProofFile(null);
      
      onSuccess?.();
      onClose();
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  const simulateOCR = () => {
    // Simulate OCR extraction
    setOcrText(`
      Invoice #12345
      Date: ${new Date().toLocaleDateString()}
      Item: Diesel Fuel
      Quantity: 500 liters
      Unit Price: $1.25
      Total: $625.00
      
      Extracted data:
      - Metric: Fuel Consumption (Diesel)
      - Value: 500
      - Unit: liters
      - Amount: $625.00
    `);
    
    // Auto-populate fields from OCR
    setMetricName('Fuel Consumption (Diesel)');
    setMetricValue('500');
    setUnit('liters');
    setAmount('625.00');
    setQuantity('500');
    setScope('Scope 1');
    setDataSource('OCR Document Scan');
  };

  const selectedMaterialData = materialMetrics.find(m => m.id === selectedMaterial);

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle>Data Entry</DialogTitle>
          <DialogDescription>
            Enter sustainability data points with multiple input methods and proof documentation
          </DialogDescription>
        </DialogHeader>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-4">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="manual">Manual Entry</TabsTrigger>
            <TabsTrigger value="bulk">Bulk Upload</TabsTrigger>
            <TabsTrigger value="ocr">OCR Scan</TabsTrigger>
            <TabsTrigger value="integration">API Integration</TabsTrigger>
          </TabsList>

          <TabsContent value="manual" className="space-y-4">
            <form onSubmit={handleManualSubmit} className="space-y-4">
              {/* Entry Type Selection */}
              <div>
                <Label htmlFor="material-select">Entry Type</Label>
                <Select
                  value={isCustomEntry ? 'custom' : selectedMaterial}
                  onValueChange={handleMaterialSelect}
                >
                  <SelectTrigger id="material-select">
                    <SelectValue placeholder="Choose entry type" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="custom">Custom Entry</SelectItem>
                    {materialMetrics.slice(0, 20).map((material) => (
                      <SelectItem key={material.id} value={material.id}>
                        <div className="flex items-center gap-2">
                          {material.category === 'Fossil Fuels' && <Flame className="h-3 w-3" />}
                          {material.category === 'Metals' && <Zap className="h-3 w-3" />}
                          {material.name}
                        </div>
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {selectedMaterialData && !isCustomEntry && (
                <Alert>
                  <AlertCircle className="h-4 w-4" />
                  <AlertDescription>
                    <strong>{selectedMaterialData.name}</strong> - {selectedMaterialData.description}
                    <br />
                    Emission Factor: {selectedMaterialData.emissionFactor} kg CO₂e/{selectedMaterialData.unit}
                    {selectedMaterialData.calorificValue && (
                      <span> | Calorific Value: {selectedMaterialData.calorificValue} MJ/{selectedMaterialData.unit}</span>
                    )}
                  </AlertDescription>
                </Alert>
              )}

              <div>
                <Label htmlFor="metric-name">Metric Name *</Label>
                <Input
                  id="metric-name"
                  value={metricName}
                  onChange={(e) => setMetricName(e.target.value)}
                  placeholder="e.g., Natural Gas Consumption, Steel Usage"
                  required
                  disabled={!isCustomEntry}
                />
                <p className="text-xs text-gray-500 mt-1">
                  Enter a clear, descriptive name for the metric being measured
                </p>
              </div>

              <div className="grid grid-cols-3 gap-4">
                <div>
                  <Label htmlFor="metric-value">Value *</Label>
                  <Input
                    id="metric-value"
                    type="number"
                    step="0.01"
                    value={metricValue}
                    onChange={(e) => setMetricValue(e.target.value)}
                    placeholder="0.00"
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="quantity">Quantity</Label>
                  <Input
                    id="quantity"
                    type="number"
                    step="0.01"
                    value={quantity}
                    onChange={(e) => setQuantity(e.target.value)}
                    placeholder="0.00"
                  />
                </div>
                <div>
                  <Label htmlFor="amount">Amount ($)</Label>
                  <Input
                    id="amount"
                    type="number"
                    step="0.01"
                    value={amount}
                    onChange={(e) => setAmount(e.target.value)}
                    placeholder="0.00"
                  />
                </div>
              </div>

              <div>
                <Label htmlFor="unit">Unit *</Label>
                {isCustomEntry ? (
                  <Select value={unit} onValueChange={setUnit}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select unit" />
                    </SelectTrigger>
                    <SelectContent>
                      {Object.entries(unitOptions).map(([category, units]) => (
                        <div key={category}>
                          <div className="px-2 py-1 text-xs font-semibold text-gray-500 uppercase">
                            {category}
                          </div>
                          {units.map((unitOption) => (
                            <SelectItem key={unitOption} value={unitOption}>
                              {unitOption}
                            </SelectItem>
                          ))}
                        </div>
                      ))}
                    </SelectContent>
                  </Select>
                ) : (
                  <Input
                    id="unit"
                    value={unit}
                    disabled
                    className="bg-gray-100"
                  />
                )}
                <p className="text-xs text-gray-500 mt-1">
                  Choose the appropriate unit of measurement for your data
                </p>
              </div>

              {selectedMaterialData && metricValue && !isCustomEntry && (
                <Alert>
                  <AlertCircle className="h-4 w-4" />
                  <AlertDescription>
                    <strong>Estimated Emissions:</strong> {' '}
                    {(parseFloat(metricValue) * selectedMaterialData.emissionFactor).toLocaleString()} kg CO₂e
                  </AlertDescription>
                </Alert>
              )}

              <div>
                <Label htmlFor="scope">GHG Scope</Label>
                <Select value={scope} onValueChange={setScope}>
                  <SelectTrigger>
                    <SelectValue placeholder="Select scope" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="Scope 1">Scope 1 - Direct Emissions</SelectItem>
                    <SelectItem value="Scope 2">Scope 2 - Indirect Energy</SelectItem>
                    <SelectItem value="Scope 3">Scope 3 - Value Chain</SelectItem>
                    <SelectItem value="not-applicable">Not Applicable</SelectItem>
                  </SelectContent>
                </Select>
                <p className="text-xs text-gray-500 mt-1">
                  Classify according to GHG Protocol standards
                </p>
              </div>

              <div>
                <Label htmlFor="reporting-period">Reporting Period</Label>
                <Input
                  id="reporting-period"
                  type="date"
                  value={reportingPeriod}
                  onChange={(e) => setReportingPeriod(e.target.value)}
                />
              </div>

              <div>
                <Label htmlFor="data-source">Data Source</Label>
                <Input
                  id="data-source"
                  value={dataSource}
                  onChange={(e) => setDataSource(e.target.value)}
                  placeholder="e.g., Utility Bills, Supplier Invoice, Internal Monitoring"
                />
              </div>

              {/* Proof Upload */}
              <div>
                <Label htmlFor="proof-upload">Upload Proof (Receipt, Voucher, LPO)</Label>
                <Input
                  id="proof-upload"
                  type="file"
                  accept=".pdf,.jpg,.jpeg,.png,.doc,.docx"
                  onChange={handleProofUpload}
                  className="mt-1"
                />
                {proofFile && (
                  <div className="flex items-center gap-2 p-2 bg-green-50 rounded-md mt-2">
                    <Receipt className="h-4 w-4 text-green-600" />
                    <span className="text-sm text-green-700">{proofFile.name}</span>
                  </div>
                )}
                <p className="text-xs text-gray-500 mt-1">
                  Upload supporting documentation for audit and verification purposes
                </p>
              </div>

              {error && (
                <Alert variant="destructive">
                  <AlertDescription>{error}</AlertDescription>
                </Alert>
              )}

              <div className="flex justify-end gap-2">
                <Button type="button" variant="outline" onClick={onClose}>
                  Cancel
                </Button>
                <Button type="submit" disabled={loading}>
                  {loading ? 'Saving...' : 'Save Data Point'}
                </Button>
              </div>
            </form>
          </TabsContent>

          <TabsContent value="bulk" className="space-y-4">
            <div>
              <Label htmlFor="file-upload">Choose File</Label>
              <Input
                id="file-upload"
                type="file"
                accept=".csv,.xlsx,.xls"
                onChange={handleFileUpload}
                className="mt-1"
              />
            </div>
            
            {file && (
              <div className="flex items-center gap-2 p-3 bg-green-50 rounded-md">
                <FileText className="h-4 w-4 text-green-600" />
                <span className="text-sm text-green-700">{file.name}</span>
              </div>
            )}

            <Alert>
              <AlertCircle className="h-4 w-4" />
              <AlertDescription>
                Supported formats: CSV, Excel (.xlsx, .xls). 
                Template available for download to ensure proper data structure.
              </AlertDescription>
            </Alert>

            <Button className="w-full" disabled={!file}>
              Process Upload
            </Button>
          </TabsContent>

          <TabsContent value="ocr" className="space-y-4">
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label htmlFor="document-upload">Upload Document</Label>
                <Input
                  id="document-upload"
                  type="file"
                  accept=".pdf,.jpg,.jpeg,.png"
                  className="mt-1"
                />
                <Button onClick={simulateOCR} className="mt-2 w-full">
                  <Scan className="h-4 w-4 mr-2" />
                  Scan Document
                </Button>
              </div>
              <div>
                <Label htmlFor="camera-capture">Camera Capture</Label>
                <Button variant="outline" className="mt-1 w-full">
                  <Camera className="h-4 w-4 mr-2" />
                  Take Photo
                </Button>
              </div>
            </div>

            {ocrText && (
              <div>
                <Label>Extracted Text</Label>
                <Textarea
                  value={ocrText}
                  onChange={(e) => setOcrText(e.target.value)}
                  rows={8}
                  className="mt-1"
                />
                <Button className="mt-2">
                  Process Extracted Data
                </Button>
              </div>
            )}
          </TabsContent>

          <TabsContent value="integration" className="space-y-4">
            <div className="text-center py-8">
              <Upload className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">API Integration</h3>
              <p className="text-gray-600 mb-4">
                Connect with external data sources, accounting systems, and IoT devices
              </p>
              <Button>Configure Integration</Button>
            </div>
          </TabsContent>
        </Tabs>
      </DialogContent>
    </Dialog>
  );
};
