
import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Upload, FileText, AlertCircle, Flame, Zap } from 'lucide-react';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { supabase } from '@/integrations/supabase/client';
import { useAuth } from '@/components/Auth/AuthProvider';
import { materialMetrics } from '@/data/materialMetrics';

interface DataUploadFormProps {
  onSuccess?: () => void;
}

export const DataUploadForm = ({ onSuccess }: DataUploadFormProps) => {
  const [file, setFile] = useState<File | null>(null);
  const [metricName, setMetricName] = useState('');
  const [metricValue, setMetricValue] = useState('');
  const [unit, setUnit] = useState('');
  const [scope, setScope] = useState('');
  const [category, setCategory] = useState('');
  const [reportingPeriod, setReportingPeriod] = useState('');
  const [dataSource, setDataSource] = useState('');
  const [selectedMaterial, setSelectedMaterial] = useState('');
  const [isCustomEntry, setIsCustomEntry] = useState(true);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const { user } = useAuth();

  const handleFileUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0];
    if (selectedFile) {
      setFile(selectedFile);
      setDataSource('File Upload');
    }
  };

  const handleMaterialSelect = (value: string) => {
    if (value === 'custom') {
      setIsCustomEntry(true);
      setSelectedMaterial('');
      setMetricName('');
      setUnit('');
      setScope('');
      setCategory('');
    } else {
      const material = materialMetrics.find(m => m.id === value);
      if (material) {
        setIsCustomEntry(false);
        setSelectedMaterial(value);
        setMetricName(material.name);
        setUnit(material.unit);
        setScope(material.scope);
        setCategory(material.category);
      }
    }
  };

  const handleManualSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!user) return;

    setLoading(true);
    setError('');

    try {
      const selectedMaterialData = materialMetrics.find(m => m.id === selectedMaterial);
      const calculatedEmissions = selectedMaterialData 
        ? parseFloat(metricValue) * selectedMaterialData.emissionFactor
        : null;

      const { error } = await supabase
        .from('esg_data_points')
        .insert([
          {
            user_id: user.id,
            metric_name: metricName,
            metric_value: parseFloat(metricValue),
            unit,
            scope,
            reporting_period: reportingPeriod,
            data_source: dataSource || 'Manual Entry',
            validation_status: 'pending'
          }
        ]);

      if (error) throw error;

      // Reset form
      setMetricName('');
      setMetricValue('');
      setUnit('');
      setScope('');
      setReportingPeriod('');
      setDataSource('');
      setSelectedMaterial('');
      setIsCustomEntry(true);
      
      onSuccess?.();
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  const selectedMaterialData = materialMetrics.find(m => m.id === selectedMaterial);

  return (
    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
      {/* Bulk Upload */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Upload className="h-5 w-5" />
            Bulk Data Upload
          </CardTitle>
          <CardDescription>
            Upload CSV or Excel files with multiple data points
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            <div>
              <Label htmlFor="file-upload">Choose File</Label>
              <Input
                id="file-upload"
                type="file"
                accept=".csv,.xlsx,.xls"
                onChange={handleFileUpload}
                className="mt-1"
              />
            </div>
            
            {file && (
              <div className="flex items-center gap-2 p-3 bg-green-50 rounded-md">
                <FileText className="h-4 w-4 text-green-600" />
                <span className="text-sm text-green-700">{file.name}</span>
              </div>
            )}

            <Alert>
              <AlertCircle className="h-4 w-4" />
              <AlertDescription>
                Supported formats: CSV, Excel (.xlsx, .xls). 
                Template available for download to ensure proper data structure.
              </AlertDescription>
            </Alert>

            <Button className="w-full" disabled={!file}>
              Process Upload
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Manual Entry */}
      <Card>
        <CardHeader>
          <CardTitle>Manual Data Entry</CardTitle>
          <CardDescription>
            Enter individual data points with AI-powered validation
          </CardDescription>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleManualSubmit} className="space-y-4">
            {/* Material Selection */}
            <div>
              <Label htmlFor="material-select">Entry Type</Label>
              <Select
                value={isCustomEntry ? 'custom' : selectedMaterial}
                onValueChange={handleMaterialSelect}
              >
                <SelectTrigger id="material-select">
                  <SelectValue placeholder="Choose entry type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="custom">Custom Entry</SelectItem>
                  {materialMetrics.slice(0, 20).map((material) => (
                    <SelectItem key={material.id} value={material.id}>
                      <div className="flex items-center gap-2">
                        {material.category === 'Fossil Fuels' && <Flame className="h-3 w-3" />}
                        {material.category === 'Metals' && <Zap className="h-3 w-3" />}
                        {material.name}
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>

            {selectedMaterialData && !isCustomEntry && (
              <Alert>
                <AlertCircle className="h-4 w-4" />
                <AlertDescription>
                  <strong>{selectedMaterialData.name}</strong> - {selectedMaterialData.description}
                  <br />
                  Emission Factor: {selectedMaterialData.emissionFactor} kg CO₂e/{selectedMaterialData.unit}
                  {selectedMaterialData.calorificValue && (
                    <span> | Calorific Value: {selectedMaterialData.calorificValue} MJ/{selectedMaterialData.unit}</span>
                  )}
                </AlertDescription>
              </Alert>
            )}

            <div>
              <Label htmlFor="metric-name">Metric Name</Label>
              <Input
                id="metric-name"
                value={metricName}
                onChange={(e) => setMetricName(e.target.value)}
                placeholder="e.g., Natural Gas Consumption, Steel Usage"
                required
                disabled={!isCustomEntry}
              />
            </div>

            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label htmlFor="metric-value">Value</Label>
                <Input
                  id="metric-value"
                  type="number"
                  step="0.01"
                  value={metricValue}
                  onChange={(e) => setMetricValue(e.target.value)}
                  placeholder="0.00"
                  required
                />
              </div>
              <div>
                <Label htmlFor="unit">Unit</Label>
                <Input
                  id="unit"
                  value={unit}
                  onChange={(e) => setUnit(e.target.value)}
                  placeholder="e.g., m³, kg, tonnes, liters"
                  disabled={!isCustomEntry}
                />
              </div>
            </div>

            {selectedMaterialData && metricValue && !isCustomEntry && (
              <Alert>
                <AlertCircle className="h-4 w-4" />
                <AlertDescription>
                  <strong>Estimated Emissions:</strong> {' '}
                  {(parseFloat(metricValue) * selectedMaterialData.emissionFactor).toLocaleString()} kg CO₂e
                </AlertDescription>
              </Alert>
            )}

            <div>
              <Label htmlFor="scope">GHG Scope</Label>
              <Select value={scope} onValueChange={setScope}>
                <SelectTrigger>
                  <SelectValue placeholder="Select scope" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="Scope 1">Scope 1 - Direct Emissions</SelectItem>
                  <SelectItem value="Scope 2">Scope 2 - Indirect Energy</SelectItem>
                  <SelectItem value="Scope 3">Scope 3 - Value Chain</SelectItem>
                  <SelectItem value="not-applicable">Not Applicable</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <Label htmlFor="reporting-period">Reporting Period</Label>
              <Input
                id="reporting-period"
                type="date"
                value={reportingPeriod}
                onChange={(e) => setReportingPeriod(e.target.value)}
              />
            </div>

            <div>
              <Label htmlFor="data-source">Data Source</Label>
              <Input
                id="data-source"
                value={dataSource}
                onChange={(e) => setDataSource(e.target.value)}
                placeholder="e.g., Utility Bills, Supplier Invoice, Internal Monitoring"
              />
            </div>

            {error && (
              <Alert variant="destructive">
                <AlertDescription>{error}</AlertDescription>
              </Alert>
            )}

            <Button type="submit" className="w-full" disabled={loading}>
              {loading ? 'Saving...' : 'Save Data Point'}
            </Button>
          </form>
        </CardContent>
      </Card>
    </div>
  );
};
