
import React, { useState, useMemo } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { 
  Pagination, 
  PaginationContent, 
  PaginationItem, 
  PaginationLink, 
  PaginationNext, 
  PaginationPrevious 
} from '@/components/ui/pagination';
import { Search, Flame, Zap, Recycle, AlertTriangle, Leaf } from 'lucide-react';
import { materialMetrics, materialCategories, MaterialData } from '@/data/materialMetrics';

export const MaterialsLibrary = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('All Materials');
  const [currentPage, setCurrentPage] = useState(1);
  const [selectedMaterial, setSelectedMaterial] = useState<MaterialData | null>(null);
  const itemsPerPage = 50;

  const filteredMaterials = useMemo(() => {
    return materialMetrics.filter(material => {
      const matchesSearch = material.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                           material.description.toLowerCase().includes(searchTerm.toLowerCase()) ||
                           material.category.toLowerCase().includes(searchTerm.toLowerCase());
      
      const matchesCategory = selectedCategory === 'All Materials' || material.category === selectedCategory;
      
      return matchesSearch && matchesCategory;
    });
  }, [searchTerm, selectedCategory]);

  const totalPages = Math.ceil(filteredMaterials.length / itemsPerPage);
  const startIndex = (currentPage - 1) * itemsPerPage;
  const paginatedMaterials = filteredMaterials.slice(startIndex, startIndex + itemsPerPage);

  const getCategoryIcon = (category: string) => {
    switch (category) {
      case 'Fossil Fuels': return <Flame className="h-4 w-4" />;
      case 'Biomass': return <Leaf className="h-4 w-4" />;
      case 'Metals': return <Zap className="h-4 w-4" />;
      case 'Waste': return <Recycle className="h-4 w-4" />;
      default: return <AlertTriangle className="h-4 w-4" />;
    }
  };

  const getScopeColor = (scope: string) => {
    switch (scope) {
      case 'Scope 1': return 'bg-red-100 text-red-800';
      case 'Scope 2': return 'bg-orange-100 text-orange-800';
      case 'Scope 3': return 'bg-blue-100 text-blue-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="text-2xl">Materials & Emission Factors Library</CardTitle>
          <div className="text-sm text-gray-600">
            Comprehensive database of {materialMetrics.length} materials with calorific values and emission factors
          </div>
        </CardHeader>
        <CardContent>
          {/* Filters */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
            <div className="relative">
              <Search className="absolute left-3 top-3 h-4 w-4 text-gray-400" />
              <Input
                placeholder="Search materials..."
                value={searchTerm}
                onChange={(e) => {
                  setSearchTerm(e.target.value);
                  setCurrentPage(1);
                }}
                className="pl-10"
              />
            </div>
            
            <Select value={selectedCategory} onValueChange={(value) => {
              setSelectedCategory(value);
              setCurrentPage(1);
            }}>
              <SelectTrigger>
                <SelectValue placeholder="Select category" />
              </SelectTrigger>
              <SelectContent>
                {materialCategories.map(category => (
                  <SelectItem key={category} value={category}>{category}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          {/* Results Summary */}
          <div className="mb-4 flex justify-between items-center">
            <p className="text-sm text-gray-600">
              Showing {startIndex + 1}-{Math.min(startIndex + itemsPerPage, filteredMaterials.length)} of {filteredMaterials.length} materials
            </p>
            <div className="text-sm text-gray-600">
              Page {currentPage} of {totalPages}
            </div>
          </div>

          {/* Materials Table */}
          <div className="border rounded-lg mb-4">
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Material Name</TableHead>
                  <TableHead>Category</TableHead>
                  <TableHead>Calorific Value</TableHead>
                  <TableHead>Emission Factor</TableHead>
                  <TableHead>Unit</TableHead>
                  <TableHead>Scope</TableHead>
                  <TableHead>Alternatives</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {paginatedMaterials.map((material) => (
                  <TableRow 
                    key={material.id}
                    className="cursor-pointer hover:bg-gray-50"
                    onClick={() => setSelectedMaterial(material)}
                  >
                    <TableCell>
                      <div>
                        <div className="font-medium">{material.name}</div>
                        <div className="text-sm text-gray-500 truncate max-w-xs">
                          {material.description}
                        </div>
                      </div>
                    </TableCell>
                    <TableCell>
                      <Badge variant="outline" className="flex items-center gap-1 w-fit">
                        {getCategoryIcon(material.category)}
                        {material.category}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      {material.calorificValue ? (
                        <span className="font-mono">
                          {material.calorificValue} MJ/{material.unit === 'm³' ? 'm³' : 'kg'}
                        </span>
                      ) : (
                        <span className="text-gray-400">N/A</span>
                      )}
                    </TableCell>
                    <TableCell>
                      <span className="font-mono">
                        {material.emissionFactor > 0 ? (
                          `${material.emissionFactor.toLocaleString()} kg CO₂e`
                        ) : material.emissionFactor < 0 ? (
                          <span className="text-green-600">
                            {Math.abs(material.emissionFactor).toLocaleString()} kg CO₂ sequestered
                          </span>
                        ) : (
                          <span className="text-green-600">Carbon neutral</span>
                        )}
                      </span>
                    </TableCell>
                    <TableCell>
                      <Badge variant="secondary">{material.unit}</Badge>
                    </TableCell>
                    <TableCell>
                      <Badge className={getScopeColor(material.scope)}>
                        {material.scope}
                      </Badge>
                    </TableCell>
                    <TableCell>
                      <div className="max-w-xs">
                        {material.alternatives ? (
                          <div className="flex flex-wrap gap-1">
                            {material.alternatives.slice(0, 2).map((alt, index) => (
                              <Badge key={index} variant="outline" className="text-xs">
                                {alt}
                              </Badge>
                            ))}
                            {material.alternatives.length > 2 && (
                              <Badge variant="outline" className="text-xs">
                                +{material.alternatives.length - 2}
                              </Badge>
                            )}
                          </div>
                        ) : (
                          <span className="text-gray-400 text-sm">None listed</span>
                        )}
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </div>

          {/* Pagination */}
          {totalPages > 1 && (
            <Pagination>
              <PaginationContent>
                <PaginationItem>
                  <PaginationPrevious 
                    onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
                    className={currentPage === 1 ? 'pointer-events-none opacity-50' : 'cursor-pointer'}
                  />
                </PaginationItem>
                
                {/* Show first page */}
                {currentPage > 3 && (
                  <>
                    <PaginationItem>
                      <PaginationLink onClick={() => setCurrentPage(1)} className="cursor-pointer">
                        1
                      </PaginationLink>
                    </PaginationItem>
                    {currentPage > 4 && <span className="px-2">...</span>}
                  </>
                )}
                
                {/* Show current page and neighbors */}
                {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                  const pageNum = Math.max(1, Math.min(totalPages - 4, currentPage - 2)) + i;
                  if (pageNum <= totalPages) {
                    return (
                      <PaginationItem key={pageNum}>
                        <PaginationLink 
                          onClick={() => setCurrentPage(pageNum)}
                          isActive={pageNum === currentPage}
                          className="cursor-pointer"
                        >
                          {pageNum}
                        </PaginationLink>
                      </PaginationItem>
                    );
                  }
                  return null;
                })}
                
                {/* Show last page */}
                {currentPage < totalPages - 2 && (
                  <>
                    {currentPage < totalPages - 3 && <span className="px-2">...</span>}
                    <PaginationItem>
                      <PaginationLink onClick={() => setCurrentPage(totalPages)} className="cursor-pointer">
                        {totalPages}
                      </PaginationLink>
                    </PaginationItem>
                  </>
                )}
                
                <PaginationItem>
                  <PaginationNext 
                    onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
                    className={currentPage === totalPages ? 'pointer-events-none opacity-50' : 'cursor-pointer'}
                  />
                </PaginationItem>
              </PaginationContent>
            </Pagination>
          )}
        </CardContent>
      </Card>

      {/* Material Details Popup */}
      {selectedMaterial && (
        <Card className="fixed inset-0 z-50 bg-black bg-opacity-50 flex items-center justify-center p-4">
          <div className="bg-white rounded-lg max-w-2xl w-full max-h-[90vh] overflow-y-auto">
            <CardHeader>
              <div className="flex justify-between items-start">
                <div>
                  <CardTitle className="flex items-center gap-2">
                    {getCategoryIcon(selectedMaterial.category)}
                    {selectedMaterial.name}
                  </CardTitle>
                  <div className="mt-2">
                    <Badge className={getScopeColor(selectedMaterial.scope)}>
                      {selectedMaterial.scope}
                    </Badge>
                  </div>
                </div>
                <Button 
                  variant="ghost" 
                  onClick={() => setSelectedMaterial(null)}
                  className="h-8 w-8 p-0"
                >
                  ×
                </Button>
              </div>
            </CardHeader>
            <CardContent className="space-y-4">
              <div>
                <h4 className="font-semibold mb-2">Description</h4>
                <p className="text-gray-700">{selectedMaterial.description}</p>
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <h4 className="font-semibold mb-2">Energy Content</h4>
                  {selectedMaterial.calorificValue ? (
                    <p className="font-mono text-lg">
                      {selectedMaterial.calorificValue} MJ/{selectedMaterial.unit === 'm³' ? 'm³' : 'kg'}
                    </p>
                  ) : (
                    <p className="text-gray-500">Not applicable</p>
                  )}
                </div>
                
                <div>
                  <h4 className="font-semibold mb-2">Emission Factor</h4>
                  <p className="font-mono text-lg">
                    {selectedMaterial.emissionFactor > 0 ? (
                      `${selectedMaterial.emissionFactor.toLocaleString()} kg CO₂e/${selectedMaterial.unit}`
                    ) : selectedMaterial.emissionFactor < 0 ? (
                      <span className="text-green-600">
                        {Math.abs(selectedMaterial.emissionFactor).toLocaleString()} kg CO₂ sequestered/{selectedMaterial.unit}
                      </span>
                    ) : (
                      <span className="text-green-600">Carbon neutral</span>
                    )}
                  </p>
                </div>
              </div>
              
              {selectedMaterial.alternatives && (
                <div>
                  <h4 className="font-semibold mb-2">Alternative Materials</h4>
                  <div className="flex flex-wrap gap-2">
                    {selectedMaterial.alternatives.map((alt, index) => (
                      <Badge key={index} variant="outline">
                        {alt}
                      </Badge>
                    ))}
                  </div>
                </div>
              )}
            </CardContent>
          </div>
        </Card>
      )}
    </div>
  );
};
