
import React from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from '@/components/ui/dialog';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { ScrollArea } from '@/components/ui/scroll-area';
import { SustainabilityMetric } from '@/data/sustainabilityMetrics';
import { 
  Calculator, 
  Database, 
  Calendar, 
  Building, 
  CheckCircle, 
  AlertCircle, 
  Target,
  Recycle,
  Globe,
  Users,
  Shield
} from 'lucide-react';

interface MetricDetailsPopupProps {
  metric: SustainabilityMetric | null;
  isOpen: boolean;
  onClose: () => void;
}

export const MetricDetailsPopup = ({ metric, isOpen, onClose }: MetricDetailsPopupProps) => {
  if (!metric) return null;

  const getCategoryIcon = (category: string) => {
    switch (category) {
      case 'Environmental': return <Globe className="h-4 w-4" />;
      case 'Social': return <Users className="h-4 w-4" />;
      case 'Governance': return <Shield className="h-4 w-4" />;
      case 'Economic': return <Target className="h-4 w-4" />;
      default: return <Database className="h-4 w-4" />;
    }
  };

  const getDifficultyColor = (difficulty: string) => {
    switch (difficulty) {
      case 'Easy': return 'bg-green-100 text-green-800';
      case 'Medium': return 'bg-yellow-100 text-yellow-800';
      case 'Hard': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-4xl max-h-[90vh]">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            {getCategoryIcon(metric.category)}
            {metric.name}
          </DialogTitle>
          <DialogDescription>
            {metric.description}
          </DialogDescription>
        </DialogHeader>

        <ScrollArea className="max-h-[70vh]">
          <div className="space-y-6">
            {/* Basic Information */}
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              <div className="space-y-2">
                <div className="text-sm font-medium text-gray-600">Category</div>
                <Badge variant="outline" className="flex items-center gap-1">
                  {getCategoryIcon(metric.category)}
                  {metric.category}
                </Badge>
              </div>
              <div className="space-y-2">
                <div className="text-sm font-medium text-gray-600">Subcategory</div>
                <Badge variant="secondary">{metric.subcategory}</Badge>
              </div>
              <div className="space-y-2">
                <div className="text-sm font-medium text-gray-600">Unit</div>
                <Badge variant="outline">{metric.unit}</Badge>
              </div>
              <div className="space-y-2">
                <div className="text-sm font-medium text-gray-600">Difficulty</div>
                <Badge className={getDifficultyColor(metric.difficulty)}>
                  {metric.difficulty}
                </Badge>
              </div>
            </div>

            {metric.scope && (
              <div className="space-y-2">
                <div className="text-sm font-medium text-gray-600">GHG Scope</div>
                <Badge variant="outline" className="bg-blue-50 text-blue-700">
                  {metric.scope}
                </Badge>
              </div>
            )}

            <Separator />

            {/* Calculation Method */}
            <Card>
              <CardHeader className="pb-3">
                <CardTitle className="text-lg flex items-center gap-2">
                  <Calculator className="h-5 w-5" />
                  Calculation Method
                </CardTitle>
              </CardHeader>
              <CardContent>
                <p className="text-sm text-gray-700">{metric.calculationMethod}</p>
              </CardContent>
            </Card>

            {/* Data Sources */}
            <Card>
              <CardHeader className="pb-3">
                <CardTitle className="text-lg flex items-center gap-2">
                  <Database className="h-5 w-5" />
                  Data Sources
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="flex flex-wrap gap-2">
                  {metric.dataSource.map((source, index) => (
                    <Badge key={index} variant="outline" className="bg-gray-50">
                      {source}
                    </Badge>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Reporting Frequency */}
            <Card>
              <CardHeader className="pb-3">
                <CardTitle className="text-lg flex items-center gap-2">
                  <Calendar className="h-5 w-5" />
                  Reporting Details
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                <div className="flex items-center gap-2">
                  <span className="text-sm font-medium">Frequency:</span>
                  <Badge variant="outline">{metric.frequency}</Badge>
                </div>
                <div className="flex items-center gap-2">
                  <span className="text-sm font-medium">Mandatory:</span>
                  {metric.mandatory ? (
                    <CheckCircle className="h-4 w-4 text-green-600" />
                  ) : (
                    <AlertCircle className="h-4 w-4 text-yellow-600" />
                  )}
                  <span className="text-sm text-gray-600">
                    {metric.mandatory ? 'Required' : 'Optional'}
                  </span>
                </div>
              </CardContent>
            </Card>

            {/* International Standards */}
            <Card>
              <CardHeader className="pb-3">
                <CardTitle className="text-lg">International Standards & Requirements</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {metric.standards.map((standard, index) => (
                    <div key={index} className="border rounded-lg p-3 bg-gray-50">
                      <div className="flex items-center gap-2 mb-2">
                        <Badge variant="outline" className="font-medium">
                          {standard.name}
                        </Badge>
                        <Badge variant="secondary" className="text-xs">
                          {standard.code}
                        </Badge>
                      </div>
                      <p className="text-sm text-gray-700">{standard.requirement}</p>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Emission Factors */}
            {metric.factors.length > 0 && (
              <Card>
                <CardHeader className="pb-3">
                  <CardTitle className="text-lg">Emission Factors & Conversion Rates</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {metric.factors.map((factor, index) => (
                      <div key={index} className="border rounded-lg p-3 bg-blue-50">
                        <div className="flex justify-between items-start mb-2">
                          <span className="font-medium text-sm">{factor.name}</span>
                          <Badge variant="outline" className="text-xs">
                            {factor.source}
                          </Badge>
                        </div>
                        <div className="text-sm text-gray-700">
                          <span className="font-medium">{factor.value}</span> {factor.unit}
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
            </Card>
            )}

            {/* Materials */}
            {metric.materials && metric.materials.length > 0 && (
              <Card>
                <CardHeader className="pb-3">
                  <CardTitle className="text-lg flex items-center gap-2">
                    <Recycle className="h-5 w-5" />
                    Associated Materials
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="flex flex-wrap gap-2">
                    {metric.materials.map((material, index) => (
                      <Badge key={index} variant="outline" className="bg-green-50 text-green-700">
                        {material}
                      </Badge>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Industry Applicability */}
            <Card>
              <CardHeader className="pb-3">
                <CardTitle className="text-lg flex items-center gap-2">
                  <Building className="h-5 w-5" />
                  Industry Applicability
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="flex flex-wrap gap-2">
                  {metric.industry.map((industry, index) => (
                    <Badge key={index} variant="outline" className="bg-purple-50 text-purple-700">
                      {industry}
                    </Badge>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </ScrollArea>
      </DialogContent>
    </Dialog>
  );
};
