
import React, { useState, useEffect } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Search, Calendar, Save } from 'lucide-react';
import { sustainabilityMetrics, SustainabilityMetric } from '@/data/sustainabilityMetrics';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface MetricSearchModalProps {
  isOpen: boolean;
  onClose: () => void;
  onSuccess: () => void;
}

export const MetricSearchModal: React.FC<MetricSearchModalProps> = ({
  isOpen,
  onClose,
  onSuccess
}) => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedMetric, setSelectedMetric] = useState<SustainabilityMetric | null>(null);
  const [metricValue, setMetricValue] = useState('');
  const [reportingPeriod, setReportingPeriod] = useState('');
  const [dataSource, setDataSource] = useState('');
  const [filteredMetrics, setFilteredMetrics] = useState<SustainabilityMetric[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const { toast } = useToast();

  useEffect(() => {
    if (searchTerm.length > 0) {
      const filtered = sustainabilityMetrics.filter(metric =>
        metric.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        metric.code.toLowerCase().includes(searchTerm.toLowerCase()) ||
        metric.category.toLowerCase().includes(searchTerm.toLowerCase())
      );
      setFilteredMetrics(filtered.slice(0, 10)); // Limit to 10 results
    } else {
      setFilteredMetrics([]);
    }
  }, [searchTerm]);

  const handleMetricSelect = (metric: SustainabilityMetric) => {
    setSelectedMetric(metric);
    setSearchTerm(metric.name);
    setFilteredMetrics([]);
  };

  const handleSave = async () => {
    if (!selectedMetric || !metricValue || !reportingPeriod) {
      toast({
        title: "Missing Information",
        description: "Please fill in all required fields",
        variant: "destructive"
      });
      return;
    }

    setIsLoading(true);
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('User not authenticated');

      const { error } = await supabase
        .from('esg_data_points')
        .insert({
          user_id: user.id,
          metric_name: selectedMetric.name,
          metric_value: parseFloat(metricValue),
          unit: selectedMetric.unit,
          reporting_period: reportingPeriod,
          data_source: dataSource || 'Manual Entry',
          validation_status: 'pending'
        });

      if (error) throw error;

      toast({
        title: "Data Point Added",
        description: `${selectedMetric.name} has been added successfully`,
      });

      // Reset form
      setSelectedMetric(null);
      setSearchTerm('');
      setMetricValue('');
      setReportingPeriod('');
      setDataSource('');
      
      onSuccess();
      onClose();
    } catch (error) {
      console.error('Error saving data point:', error);
      toast({
        title: "Error",
        description: "Failed to save data point",
        variant: "destructive"
      });
    } finally {
      setIsLoading(false);
    }
  };

  const resetForm = () => {
    setSelectedMetric(null);
    setSearchTerm('');
    setMetricValue('');
    setReportingPeriod('');
    setDataSource('');
    setFilteredMetrics([]);
  };

  const handleClose = () => {
    resetForm();
    onClose();
  };

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="max-w-2xl">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Search className="h-5 w-5" />
            Add ESG Data Point
          </DialogTitle>
        </DialogHeader>

        <div className="space-y-4">
          {/* Metric Search */}
          <div className="relative">
            <Label htmlFor="metric-search">Search Metric</Label>
            <Input
              id="metric-search"
              placeholder="Search by metric name, code, or category..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="mt-1"
            />
            
            {filteredMetrics.length > 0 && (
              <div className="absolute z-10 w-full mt-1 bg-white border rounded-md shadow-lg max-h-60 overflow-auto">
                {filteredMetrics.map((metric) => (
                  <div
                    key={metric.id}
                    className="p-3 hover:bg-gray-50 cursor-pointer border-b"
                    onClick={() => handleMetricSelect(metric)}
                  >
                    <div className="flex justify-between items-start">
                      <div>
                        <p className="font-medium text-sm">{metric.name}</p>
                        <p className="text-xs text-gray-500">{metric.code} • {metric.category}</p>
                        <p className="text-xs text-gray-600 mt-1">{metric.description}</p>
                      </div>
                      <Badge variant="outline" className="text-xs">{metric.unit}</Badge>
                    </div>
                    <div className="flex flex-wrap gap-1 mt-2">
                      {Object.entries(metric.standardCodes).map(([standard, codes]) => (
                        codes.map((code, index) => (
                          <Badge key={`${standard}-${index}`} variant="secondary" className="text-xs">
                            {standard} {code}
                          </Badge>
                        ))
                      ))}
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>

          {/* Selected Metric Details */}
          {selectedMetric && (
            <div className="p-4 bg-blue-50 rounded-lg">
              <h4 className="font-semibold text-blue-900">{selectedMetric.name}</h4>
              <p className="text-sm text-blue-700 mt-1">{selectedMetric.description}</p>
              <div className="flex flex-wrap gap-1 mt-2">
                {Object.entries(selectedMetric.standardCodes).map(([standard, codes]) => (
                  codes.map((code, index) => (
                    <Badge key={`${standard}-${index}`} variant="secondary" className="text-xs">
                      {standard} {code}
                    </Badge>
                  ))
                ))}
              </div>
            </div>
          )}

          {/* Data Entry Form */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <Label htmlFor="metric-value">Value *</Label>
              <Input
                id="metric-value"
                type="number"
                placeholder="Enter value"
                value={metricValue}
                onChange={(e) => setMetricValue(e.target.value)}
                className="mt-1"
              />
              {selectedMetric && (
                <p className="text-xs text-gray-500 mt-1">Unit: {selectedMetric.unit}</p>
              )}
            </div>
            
            <div>
              <Label htmlFor="reporting-period">Reporting Period *</Label>
              <Input
                id="reporting-period"
                type="date"
                value={reportingPeriod}
                onChange={(e) => setReportingPeriod(e.target.value)}
                className="mt-1"
              />
            </div>
          </div>

          <div>
            <Label htmlFor="data-source">Data Source</Label>
            <Input
              id="data-source"
              placeholder="e.g., Internal systems, Third-party audit, Manual calculation"
              value={dataSource}
              onChange={(e) => setDataSource(e.target.value)}
              className="mt-1"
            />
          </div>

          {/* Action Buttons */}
          <div className="flex justify-end gap-2 pt-4">
            <Button variant="outline" onClick={handleClose}>
              Cancel
            </Button>
            <Button 
              onClick={handleSave} 
              disabled={!selectedMetric || !metricValue || !reportingPeriod || isLoading}
              className="flex items-center gap-2"
            >
              <Save className="h-4 w-4" />
              {isLoading ? 'Saving...' : 'Save Data Point'}
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
};
