
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { supabase } from '@/integrations/supabase/client';
import { useAuth } from '@/components/Auth/AuthProvider';
import { 
  Users, 
  Activity, 
  Target, 
  Search, 
  Filter, 
  Edit, 
  Trash2, 
  UserPlus,
  Settings,
  BarChart3,
  CheckCircle2,
  Clock,
  AlertTriangle
} from 'lucide-react';

interface DataPoint {
  id: string;
  metric_name: string;
  metric_value: number;
  unit: string;
  scope: string;
  reporting_period: string;
  data_source: string;
  validation_status: string;
  created_at: string;
}

interface OrganizationStats {
  totalDataPoints: number;
  validatedPoints: number;
  materialsTracked: number;
  carbonFootprint: number;
  monthlyGrowth: number;
  validationRate: number;
}

export const OrganizationDataManager = () => {
  const [data, setData] = useState<DataPoint[]>([]);
  const [stats, setStats] = useState<OrganizationStats>({
    totalDataPoints: 0,
    validatedPoints: 0,
    materialsTracked: 0,
    carbonFootprint: 0,
    monthlyGrowth: 0,
    validationRate: 0
  });
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterScope, setFilterScope] = useState('all');
  const [filterStatus, setFilterStatus] = useState('all');
  const [filterPeriod, setFilterPeriod] = useState('all');
  const { user } = useAuth();

  useEffect(() => {
    if (user) {
      fetchOrganizationData();
      fetchOrganizationStats();
    }
  }, [user]);

  const fetchOrganizationData = async () => {
    try {
      const { data: dataPoints, error } = await supabase
        .from('esg_data_points')
        .select('*')
        .eq('user_id', user?.id)
        .order('created_at', { ascending: false });

      if (error) throw error;
      setData(dataPoints || []);
    } catch (error) {
      console.error('Error fetching organization data:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchOrganizationStats = async () => {
    try {
      const { data: dataPoints, error } = await supabase
        .from('esg_data_points')
        .select('*')
        .eq('user_id', user?.id);

      if (error) throw error;

      const total = dataPoints?.length || 0;
      const validated = dataPoints?.filter(dp => dp.validation_status === 'validated').length || 0;
      const materials = new Set(dataPoints?.map(dp => dp.metric_name)).size;
      const emissions = dataPoints?.reduce((sum, dp) => sum + (dp.metric_value || 0), 0) || 0;

      setStats({
        totalDataPoints: total,
        validatedPoints: validated,
        materialsTracked: materials,
        carbonFootprint: Math.round(emissions * 2.5), // Rough emission factor
        monthlyGrowth: 12, // This would be calculated based on historical data
        validationRate: total > 0 ? Math.round((validated / total) * 100) : 0
      });
    } catch (error) {
      console.error('Error fetching stats:', error);
    }
  };

  const filteredData = data.filter(item => {
    const matchesSearch = item.metric_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         item.data_source.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesScope = filterScope === 'all' || item.scope === filterScope;
    const matchesStatus = filterStatus === 'all' || item.validation_status === filterStatus;
    const matchesPeriod = filterPeriod === 'all' || 
      (filterPeriod === 'current-month' && new Date(item.reporting_period).getMonth() === new Date().getMonth());
    
    return matchesSearch && matchesScope && matchesStatus && matchesPeriod;
  });

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'validated': return 'bg-green-100 text-green-800';
      case 'flagged': return 'bg-red-100 text-red-800';
      default: return 'bg-yellow-100 text-yellow-800';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'validated': return <CheckCircle2 className="h-3 w-3" />;
      case 'flagged': return <AlertTriangle className="h-3 w-3" />;
      default: return <Clock className="h-3 w-3" />;
    }
  };

  if (loading) {
    return <div className="p-6">Loading organization data...</div>;
  }

  return (
    <div className="space-y-6">
      {/* Organization Overview Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-gray-600 flex items-center gap-2">
              <Activity className="h-4 w-4" />
              Total Data Points
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.totalDataPoints.toLocaleString()}</div>
            <div className="text-xs text-green-600">+{stats.monthlyGrowth}% from last month</div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-gray-600 flex items-center gap-2">
              <CheckCircle2 className="h-4 w-4" />
              Validated Data
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.validatedPoints.toLocaleString()}</div>
            <div className="text-xs text-green-600">{stats.validationRate}% validation rate</div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-gray-600 flex items-center gap-2">
              <Target className="h-4 w-4" />
              Materials Tracked
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.materialsTracked}</div>
            <div className="text-xs text-blue-600">Unique metrics monitored</div>
          </CardContent>
        </Card>
        
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-gray-600 flex items-center gap-2">
              <BarChart3 className="h-4 w-4" />
              Carbon Footprint
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{stats.carbonFootprint.toLocaleString()}</div>
            <div className="text-xs text-red-600">tCO₂e this period</div>
          </CardContent>
        </Card>
      </div>

      <Tabs defaultValue="data-points" className="space-y-4">
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="data-points">Data Points</TabsTrigger>
          <TabsTrigger value="team-management">Team</TabsTrigger>
          <TabsTrigger value="processes">Processes</TabsTrigger>
          <TabsTrigger value="integrations">Integrations</TabsTrigger>
        </TabsList>

        <TabsContent value="data-points" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>Organization Data Points</CardTitle>
              
              {/* Filters */}
              <div className="flex flex-wrap gap-4 items-center">
                <div className="flex items-center gap-2">
                  <Search className="h-4 w-4 text-gray-500" />
                  <Input
                    placeholder="Search data points..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-64"
                  />
                </div>
                
                <Select value={filterScope} onValueChange={setFilterScope}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Filter by scope" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Scopes</SelectItem>
                    <SelectItem value="Scope 1">Scope 1</SelectItem>
                    <SelectItem value="Scope 2">Scope 2</SelectItem>
                    <SelectItem value="Scope 3">Scope 3</SelectItem>
                  </SelectContent>
                </Select>

                <Select value={filterStatus} onValueChange={setFilterStatus}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Filter by status" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Status</SelectItem>
                    <SelectItem value="pending">Pending</SelectItem>
                    <SelectItem value="validated">Validated</SelectItem>
                    <SelectItem value="flagged">Flagged</SelectItem>
                  </SelectContent>
                </Select>

                <Select value={filterPeriod} onValueChange={setFilterPeriod}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Filter by period" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Periods</SelectItem>
                    <SelectItem value="current-month">This Month</SelectItem>
                    <SelectItem value="last-month">Last Month</SelectItem>
                    <SelectItem value="current-quarter">This Quarter</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </CardHeader>
            
            <CardContent>
              {filteredData.length === 0 ? (
                <Alert>
                  <AlertDescription>
                    No data points found matching your criteria. Use the data entry modal to add your first ESG metrics.
                  </AlertDescription>
                </Alert>
              ) : (
                <Table>
                  <TableHeader>
                    <TableRow>
                      <TableHead>Metric</TableHead>
                      <TableHead>Value</TableHead>
                      <TableHead>Unit</TableHead>
                      <TableHead>Scope</TableHead>
                      <TableHead>Period</TableHead>
                      <TableHead>Source</TableHead>
                      <TableHead>Status</TableHead>
                      <TableHead>Actions</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {filteredData.map((item) => (
                      <TableRow key={item.id}>
                        <TableCell className="font-medium">{item.metric_name}</TableCell>
                        <TableCell>{item.metric_value?.toLocaleString()}</TableCell>
                        <TableCell>{item.unit}</TableCell>
                        <TableCell>
                          <Badge variant="outline">
                            {item.scope?.replace('-', ' ') || 'N/A'}
                          </Badge>
                        </TableCell>
                        <TableCell>{new Date(item.reporting_period).toLocaleDateString()}</TableCell>
                        <TableCell>{item.data_source}</TableCell>
                        <TableCell>
                          <Badge className={getStatusColor(item.validation_status)}>
                            <div className="flex items-center gap-1">
                              {getStatusIcon(item.validation_status)}
                              {item.validation_status}
                            </div>
                          </Badge>
                        </TableCell>
                        <TableCell>
                          <div className="flex gap-2">
                            <Button variant="ghost" size="sm">
                              <Edit className="h-4 w-4" />
                            </Button>
                            <Button variant="ghost" size="sm">
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="team-management" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Users className="h-5 w-5" />
                Sustainability Team Management
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex justify-between items-center">
                  <p className="text-sm text-gray-600">
                    Manage sustainability officers and their access permissions
                  </p>
                  <Button className="flex items-center gap-2">
                    <UserPlus className="h-4 w-4" />
                    Add Team Member
                  </Button>
                </div>
                
                <Alert>
                  <Users className="h-4 w-4" />
                  <AlertDescription>
                    Multiple sustainability officers can contribute data from different departments 
                    and processes within your organization. Each member's contributions are tracked 
                    and validated separately.
                  </AlertDescription>
                </Alert>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="processes" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Settings className="h-5 w-5" />
                Organizational Processes
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Alert>
                <Settings className="h-4 w-4" />
                <AlertDescription>
                  Configure and manage your organization's sustainability processes and procedures. 
                  This includes defining data collection workflows, validation rules, and reporting schedules.
                </AlertDescription>
              </Alert>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="integrations" className="space-y-4">
          <Card>
            <CardHeader>
              <CardTitle>External Data Integrations</CardTitle>
            </CardHeader>
            <CardContent>
              <Alert>
                <Activity className="h-4 w-4" />
                <AlertDescription>
                  Connect external data sources such as utility providers, accounting systems, 
                  and IoT sensors to automatically import sustainability data.
                </AlertDescription>
              </Alert>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
};
