
import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Switch } from '@/components/ui/switch';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import { Separator } from '@/components/ui/separator';
import { Badge } from '@/components/ui/badge';
import { Settings, Save, Globe, Calendar, Ruler, Thermometer } from 'lucide-react';

interface SystemSettings {
  reportingFrequency: string;
  linearUnit: string;
  areaUnit: string;
  temperatureUnit: string;
  currencyUnit: string;
  energyUnit: string;
  timezone: string;
  dateFormat: string;
  numberFormat: string;
  autoValidation: boolean;
  aiAssistance: boolean;
  realTimeSync: boolean;
}

export const SystemSettings = () => {
  const [settings, setSettings] = useState<SystemSettings>({
    reportingFrequency: 'quarterly',
    linearUnit: 'meters',
    areaUnit: 'square-meters',
    temperatureUnit: 'celsius',
    currencyUnit: 'usd',
    energyUnit: 'kwh',
    timezone: 'utc',
    dateFormat: 'dd/mm/yyyy',
    numberFormat: 'comma',
    autoValidation: true,
    aiAssistance: true,
    realTimeSync: false
  });

  const [hasChanges, setHasChanges] = useState(false);

  const updateSetting = (key: keyof SystemSettings, value: any) => {
    setSettings(prev => ({ ...prev, [key]: value }));
    setHasChanges(true);
  };

  const saveSettings = () => {
    // Here you would typically save to your backend
    console.log('Saving settings:', settings);
    setHasChanges(false);
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Settings className="h-5 w-5" />
            System Settings
          </CardTitle>
          <div className="text-sm text-gray-600">
            Configure global settings that apply across the entire application
          </div>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Reporting Settings */}
          <div>
            <h3 className="text-lg font-medium mb-4 flex items-center gap-2">
              <Calendar className="h-4 w-4" />
              Reporting Configuration
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label htmlFor="reporting-frequency">Default Reporting Frequency</Label>
                <Select
                  value={settings.reportingFrequency}
                  onValueChange={(value) => updateSetting('reportingFrequency', value)}
                >
                  <SelectTrigger id="reporting-frequency">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="monthly">Monthly</SelectItem>
                    <SelectItem value="quarterly">Quarterly</SelectItem>
                    <SelectItem value="semi-annually">Semi-Annually</SelectItem>
                    <SelectItem value="annually">Annually</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="date-format">Date Format</Label>
                <Select
                  value={settings.dateFormat}
                  onValueChange={(value) => updateSetting('dateFormat', value)}
                >
                  <SelectTrigger id="date-format">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="dd/mm/yyyy">DD/MM/YYYY</SelectItem>
                    <SelectItem value="mm/dd/yyyy">MM/DD/YYYY</SelectItem>
                    <SelectItem value="yyyy-mm-dd">YYYY-MM-DD</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </div>

          <Separator />

          {/* Unit Settings */}
          <div>
            <h3 className="text-lg font-medium mb-4 flex items-center gap-2">
              <Ruler className="h-4 w-4" />
              Unit Preferences
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div>
                <Label htmlFor="linear-unit">Linear Measurement</Label>
                <Select
                  value={settings.linearUnit}
                  onValueChange={(value) => updateSetting('linearUnit', value)}
                >
                  <SelectTrigger id="linear-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="meters">Meters (m)</SelectItem>
                    <SelectItem value="feet">Feet (ft)</SelectItem>
                    <SelectItem value="inches">Inches (in)</SelectItem>
                    <SelectItem value="kilometers">Kilometers (km)</SelectItem>
                    <SelectItem value="miles">Miles (mi)</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="area-unit">Area Measurement</Label>
                <Select
                  value={settings.areaUnit}
                  onValueChange={(value) => updateSetting('areaUnit', value)}
                >
                  <SelectTrigger id="area-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="square-meters">Square Meters (m²)</SelectItem>
                    <SelectItem value="square-feet">Square Feet (ft²)</SelectItem>
                    <SelectItem value="hectares">Hectares (ha)</SelectItem>
                    <SelectItem value="acres">Acres</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="temperature-unit">Temperature</Label>
                <Select
                  value={settings.temperatureUnit}
                  onValueChange={(value) => updateSetting('temperatureUnit', value)}
                >
                  <SelectTrigger id="temperature-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="celsius">Celsius (°C)</SelectItem>
                    <SelectItem value="fahrenheit">Fahrenheit (°F)</SelectItem>
                    <SelectItem value="kelvin">Kelvin (K)</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="energy-unit">Energy Unit</Label>
                <Select
                  value={settings.energyUnit}
                  onValueChange={(value) => updateSetting('energyUnit', value)}
                >
                  <SelectTrigger id="energy-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="kwh">Kilowatt Hours (kWh)</SelectItem>
                    <SelectItem value="mwh">Megawatt Hours (MWh)</SelectItem>
                    <SelectItem value="gwh">Gigawatt Hours (GWh)</SelectItem>
                    <SelectItem value="mj">Megajoules (MJ)</SelectItem>
                    <SelectItem value="gj">Gigajoules (GJ)</SelectItem>
                    <SelectItem value="btu">British Thermal Units (BTU)</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="currency-unit">Currency</Label>
                <Select
                  value={settings.currencyUnit}
                  onValueChange={(value) => updateSetting('currencyUnit', value)}
                >
                  <SelectTrigger id="currency-unit">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="usd">US Dollar (USD)</SelectItem>
                    <SelectItem value="eur">Euro (EUR)</SelectItem>
                    <SelectItem value="gbp">British Pound (GBP)</SelectItem>
                    <SelectItem value="jpy">Japanese Yen (JPY)</SelectItem>
                    <SelectItem value="cad">Canadian Dollar (CAD)</SelectItem>
                    <SelectItem value="aud">Australian Dollar (AUD)</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label htmlFor="number-format">Number Format</Label>
                <Select
                  value={settings.numberFormat}
                  onValueChange={(value) => updateSetting('numberFormat', value)}
                >
                  <SelectTrigger id="number-format">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="comma">1,234.56</SelectItem>
                    <SelectItem value="space">1 234.56</SelectItem>
                    <SelectItem value="period">1.234,56</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </div>

          <Separator />

          {/* System Preferences */}
          <div>
            <h3 className="text-lg font-medium mb-4 flex items-center gap-2">
              <Globe className="h-4 w-4" />
              System Preferences
            </h3>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <div>
                  <Label htmlFor="auto-validation">Automatic Data Validation</Label>
                  <p className="text-sm text-gray-600">Enable AI-powered validation of data entries</p>
                </div>
                <Switch
                  id="auto-validation"
                  checked={settings.autoValidation}
                  onCheckedChange={(checked) => updateSetting('autoValidation', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div>
                  <Label htmlFor="ai-assistance">AI Assistance</Label>
                  <p className="text-sm text-gray-600">Enable AI suggestions and gap-filling</p>
                </div>
                <Switch
                  id="ai-assistance"
                  checked={settings.aiAssistance}
                  onCheckedChange={(checked) => updateSetting('aiAssistance', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div>
                  <Label htmlFor="real-time-sync">Real-time Synchronization</Label>
                  <p className="text-sm text-gray-600">Sync data across all devices in real-time</p>
                </div>
                <Switch
                  id="real-time-sync"
                  checked={settings.realTimeSync}
                  onCheckedChange={(checked) => updateSetting('realTimeSync', checked)}
                />
              </div>
            </div>
          </div>

          <Separator />

          {/* Current Settings Preview */}
          <div>
            <h3 className="text-lg font-medium mb-4">Current Configuration</h3>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              <div>
                <Label className="text-xs text-gray-500">REPORTING</Label>
                <Badge variant="outline" className="mt-1 capitalize">
                  {settings.reportingFrequency.replace('-', ' ')}
                </Badge>
              </div>
              <div>
                <Label className="text-xs text-gray-500">LINEAR UNIT</Label>
                <Badge variant="outline" className="mt-1 capitalize">
                  {settings.linearUnit}
                </Badge>
              </div>
              <div>
                <Label className="text-xs text-gray-500">AREA UNIT</Label>
                <Badge variant="outline" className="mt-1 capitalize">
                  {settings.areaUnit.replace('-', ' ')}
                </Badge>
              </div>
              <div>
                <Label className="text-xs text-gray-500">TEMPERATURE</Label>
                <Badge variant="outline" className="mt-1 capitalize">
                  {settings.temperatureUnit}
                </Badge>
              </div>
            </div>
          </div>

          {/* Save Button */}
          <div className="flex justify-end">
            <Button 
              onClick={saveSettings} 
              disabled={!hasChanges}
              className="flex items-center gap-2"
            >
              <Save className="h-4 w-4" />
              Save Settings
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
