
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { Plus, Edit, Trash2, Zap } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface Initiative {
  id: string;
  target_id: string;
  initiative_name: string;
  initiative_type: string;
  description: string;
  estimated_reduction: number;
  investment_required: number;
  currency: string;
  implementation_date: string;
  completion_date: string;
  status: string;
  actual_reduction: number;
  roi_percentage: number;
  responsible_team: string;
  priority_level: string;
}

interface NetZeroTarget {
  id: string;
  target_name: string;
}

interface InitiativeManagerProps {
  onUpdate: () => void;
}

export const InitiativeManager: React.FC<InitiativeManagerProps> = ({ onUpdate }) => {
  const [initiatives, setInitiatives] = useState<Initiative[]>([]);
  const [targets, setTargets] = useState<NetZeroTarget[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [showForm, setShowForm] = useState(false);
  const [editingInitiative, setEditingInitiative] = useState<Initiative | null>(null);
  const [formData, setFormData] = useState({
    target_id: '',
    initiative_name: '',
    initiative_type: 'energy_efficiency',
    description: '',
    estimated_reduction: '',
    investment_required: '',
    currency: 'USD',
    implementation_date: '',
    completion_date: '',
    status: 'planned',
    actual_reduction: '',
    roi_percentage: '',
    responsible_team: '',
    priority_level: 'medium'
  });
  const { toast } = useToast();

  useEffect(() => {
    fetchData();
  }, []);

  const fetchData = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      // Fetch initiatives
      const { data: initiativesData, error: initiativesError } = await supabase
        .from('decarbonization_initiatives')
        .select('*')
        .eq('user_id', user.id)
        .order('created_at', { ascending: false });

      if (initiativesError) throw initiativesError;

      // Fetch targets
      const { data: targetsData, error: targetsError } = await supabase
        .from('net_zero_targets')
        .select('id, target_name')
        .eq('user_id', user.id);

      if (targetsError) throw targetsError;

      setInitiatives(initiativesData || []);
      setTargets(targetsData || []);
    } catch (error) {
      console.error('Error fetching data:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('User not authenticated');

      const initiativeData = {
        user_id: user.id,
        target_id: formData.target_id || null,
        initiative_name: formData.initiative_name,
        initiative_type: formData.initiative_type,
        description: formData.description,
        estimated_reduction: parseFloat(formData.estimated_reduction),
        investment_required: formData.investment_required ? parseFloat(formData.investment_required) : null,
        currency: formData.currency,
        implementation_date: formData.implementation_date || null,
        completion_date: formData.completion_date || null,
        status: formData.status,
        actual_reduction: formData.actual_reduction ? parseFloat(formData.actual_reduction) : null,
        roi_percentage: formData.roi_percentage ? parseFloat(formData.roi_percentage) : null,
        responsible_team: formData.responsible_team,
        priority_level: formData.priority_level
      };

      if (editingInitiative) {
        const { error } = await supabase
          .from('decarbonization_initiatives')
          .update(initiativeData)
          .eq('id', editingInitiative.id);
        
        if (error) throw error;
        toast({ title: "Initiative updated successfully" });
      } else {
        const { error } = await supabase
          .from('decarbonization_initiatives')
          .insert(initiativeData);
        
        if (error) throw error;
        toast({ title: "Initiative added successfully" });
      }

      resetForm();
      fetchData();
      onUpdate();
    } catch (error) {
      console.error('Error saving initiative:', error);
      toast({
        title: "Error saving initiative",
        description: "Please try again",
        variant: "destructive"
      });
    }
  };

  const resetForm = () => {
    setFormData({
      target_id: '',
      initiative_name: '',
      initiative_type: 'energy_efficiency',
      description: '',
      estimated_reduction: '',
      investment_required: '',
      currency: 'USD',
      implementation_date: '',
      completion_date: '',
      status: 'planned',
      actual_reduction: '',
      roi_percentage: '',
      responsible_team: '',
      priority_level: 'medium'
    });
    setEditingInitiative(null);
    setShowForm(false);
  };

  const handleEdit = (initiative: Initiative) => {
    setFormData({
      target_id: initiative.target_id || '',
      initiative_name: initiative.initiative_name,
      initiative_type: initiative.initiative_type,
      description: initiative.description || '',
      estimated_reduction: initiative.estimated_reduction.toString(),
      investment_required: initiative.investment_required?.toString() || '',
      currency: initiative.currency,
      implementation_date: initiative.implementation_date || '',
      completion_date: initiative.completion_date || '',
      status: initiative.status,
      actual_reduction: initiative.actual_reduction?.toString() || '',
      roi_percentage: initiative.roi_percentage?.toString() || '',
      responsible_team: initiative.responsible_team || '',
      priority_level: initiative.priority_level
    });
    setEditingInitiative(initiative);
    setShowForm(true);
  };

  const handleDelete = async (id: string) => {
    try {
      const { error } = await supabase
        .from('decarbonization_initiatives')
        .delete()
        .eq('id', id);

      if (error) throw error;
      
      toast({ title: "Initiative deleted successfully" });
      fetchData();
      onUpdate();
    } catch (error) {
      console.error('Error deleting initiative:', error);
      toast({
        title: "Error deleting initiative",
        variant: "destructive"
      });
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'planned': return 'bg-gray-100 text-gray-800';
      case 'in_progress': return 'bg-yellow-100 text-yellow-800';
      case 'completed': return 'bg-green-100 text-green-800';
      case 'cancelled': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high': return 'bg-red-100 text-red-800';
      case 'medium': return 'bg-yellow-100 text-yellow-800';
      case 'low': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  if (isLoading) {
    return <div className="animate-pulse h-64 bg-gray-200 rounded"></div>;
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold">Decarbonization Initiatives</h3>
        <Button onClick={() => setShowForm(true)} className="flex items-center gap-2">
          <Plus className="h-4 w-4" />
          Add Initiative
        </Button>
      </div>

      {showForm && (
        <Card>
          <CardHeader>
            <CardTitle>
              {editingInitiative ? 'Edit Initiative' : 'Add New Initiative'}
            </CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <Label>Initiative Name *</Label>
                  <Input
                    value={formData.initiative_name}
                    onChange={(e) => setFormData({...formData, initiative_name: e.target.value})}
                    required
                  />
                </div>
                <div>
                  <Label>Related Target</Label>
                  <Select value={formData.target_id} onValueChange={(value) => setFormData({...formData, target_id: value})}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select a target" />
                    </SelectTrigger>
                    <SelectContent>
                      {targets.map((target) => (
                        <SelectItem key={target.id} value={target.id}>
                          {target.target_name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Initiative Type *</Label>
                  <Select value={formData.initiative_type} onValueChange={(value) => setFormData({...formData, initiative_type: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="energy_efficiency">Energy Efficiency</SelectItem>
                      <SelectItem value="renewable_energy">Renewable Energy</SelectItem>
                      <SelectItem value="process_improvement">Process Improvement</SelectItem>
                      <SelectItem value="fuel_switching">Fuel Switching</SelectItem>
                      <SelectItem value="carbon_capture">Carbon Capture</SelectItem>
                      <SelectItem value="nature_based">Nature-Based Solutions</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Priority Level</Label>
                  <Select value={formData.priority_level} onValueChange={(value) => setFormData({...formData, priority_level: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="high">High</SelectItem>
                      <SelectItem value="medium">Medium</SelectItem>
                      <SelectItem value="low">Low</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Estimated Reduction (tCO2e/year) *</Label>
                  <Input
                    type="number"
                    step="0.01"
                    value={formData.estimated_reduction}
                    onChange={(e) => setFormData({...formData, estimated_reduction: e.target.value})}
                    required
                  />
                </div>
                <div>
                  <Label>Investment Required</Label>
                  <Input
                    type="number"
                    step="0.01"
                    value={formData.investment_required}
                    onChange={(e) => setFormData({...formData, investment_required: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Currency</Label>
                  <Select value={formData.currency} onValueChange={(value) => setFormData({...formData, currency: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="USD">USD</SelectItem>
                      <SelectItem value="EUR">EUR</SelectItem>
                      <SelectItem value="GBP">GBP</SelectItem>
                      <SelectItem value="JPY">JPY</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Status</Label>
                  <Select value={formData.status} onValueChange={(value) => setFormData({...formData, status: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="planned">Planned</SelectItem>
                      <SelectItem value="in_progress">In Progress</SelectItem>
                      <SelectItem value="completed">Completed</SelectItem>
                      <SelectItem value="cancelled">Cancelled</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Implementation Date</Label>
                  <Input
                    type="date"
                    value={formData.implementation_date}
                    onChange={(e) => setFormData({...formData, implementation_date: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Completion Date</Label>
                  <Input
                    type="date"
                    value={formData.completion_date}
                    onChange={(e) => setFormData({...formData, completion_date: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Actual Reduction (tCO2e/year)</Label>
                  <Input
                    type="number"
                    step="0.01"
                    value={formData.actual_reduction}
                    onChange={(e) => setFormData({...formData, actual_reduction: e.target.value})}
                  />
                </div>
                <div>
                  <Label>ROI Percentage</Label>
                  <Input
                    type="number"
                    step="0.01"
                    value={formData.roi_percentage}
                    onChange={(e) => setFormData({...formData, roi_percentage: e.target.value})}
                  />
                </div>
              </div>
              
              <div>
                <Label>Responsible Team</Label>
                <Input
                  value={formData.responsible_team}
                  onChange={(e) => setFormData({...formData, responsible_team: e.target.value})}
                />
              </div>

              <div>
                <Label>Description</Label>
                <Textarea
                  value={formData.description}
                  onChange={(e) => setFormData({...formData, description: e.target.value})}
                  rows={3}
                />
              </div>

              <div className="flex gap-2">
                <Button type="submit">
                  {editingInitiative ? 'Update Initiative' : 'Add Initiative'}
                </Button>
                <Button type="button" variant="outline" onClick={resetForm}>
                  Cancel
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>
      )}

      <div className="grid grid-cols-1 gap-4">
        {initiatives.map((initiative) => (
          <Card key={initiative.id}>
            <CardContent className="p-4">
              <div className="flex justify-between items-start">
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-2">
                    <h4 className="font-semibold">{initiative.initiative_name}</h4>
                    <Badge className={getStatusColor(initiative.status)}>
                      {initiative.status}
                    </Badge>
                    <Badge className={getPriorityColor(initiative.priority_level)}>
                      {initiative.priority_level}
                    </Badge>
                  </div>
                  
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm mb-3">
                    <div>
                      <p className="text-gray-600">Type</p>
                      <p className="font-medium">{initiative.initiative_type.replace('_', ' ')}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Est. Reduction</p>
                      <p className="font-medium">{initiative.estimated_reduction.toFixed(1)} tCO2e/year</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Investment</p>
                      <p className="font-medium">
                        {initiative.investment_required 
                          ? `${initiative.investment_required.toLocaleString()} ${initiative.currency}`
                          : 'N/A'
                        }
                      </p>
                    </div>
                    <div>
                      <p className="text-gray-600">ROI</p>
                      <p className="font-medium">
                        {initiative.roi_percentage ? `${initiative.roi_percentage.toFixed(1)}%` : 'N/A'}
                      </p>
                    </div>
                  </div>

                  {initiative.description && (
                    <p className="text-sm text-gray-600 mb-2">{initiative.description}</p>
                  )}

                  {initiative.responsible_team && (
                    <div className="text-sm">
                      <span className="text-gray-600">Team: </span>
                      <span className="font-medium">{initiative.responsible_team}</span>
                    </div>
                  )}
                </div>
                
                <div className="flex gap-2">
                  <Button size="sm" variant="outline" onClick={() => handleEdit(initiative)}>
                    <Edit className="h-4 w-4" />
                  </Button>
                  <Button size="sm" variant="outline" onClick={() => handleDelete(initiative.id)}>
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {initiatives.length === 0 && (
        <Card>
          <CardContent className="p-8 text-center">
            <Zap className="h-12 w-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-600 mb-2">No initiatives yet</h3>
            <p className="text-gray-500 mb-4">Start planning your decarbonization journey by adding your first initiative.</p>
            <Button onClick={() => setShowForm(true)}>
              Add First Initiative
            </Button>
          </CardContent>
        </Card>
      )}
    </div>
  );
};
