
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Target, TrendingDown, Calendar, CheckCircle, AlertTriangle } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';

interface NetZeroSummary {
  totalTargets: number;
  activeTargets: number;
  achievedTargets: number;
  atRiskTargets: number;
  totalInitiatives: number;
  plannedInitiatives: number;
  inProgressInitiatives: number;
  completedInitiatives: number;
  totalReductionPotential: number;
  avgProgress: number;
}

export const NetZeroOverview: React.FC = () => {
  const [summary, setSummary] = useState<NetZeroSummary>({
    totalTargets: 0,
    activeTargets: 0,
    achievedTargets: 0,
    atRiskTargets: 0,
    totalInitiatives: 0,
    plannedInitiatives: 0,
    inProgressInitiatives: 0,
    completedInitiatives: 0,
    totalReductionPotential: 0,
    avgProgress: 0
  });
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    fetchNetZeroSummary();
  }, []);

  const fetchNetZeroSummary = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      // Fetch targets
      const { data: targets, error: targetsError } = await supabase
        .from('net_zero_targets')
        .select('*')
        .eq('user_id', user.id);

      if (targetsError) throw targetsError;

      // Fetch initiatives
      const { data: initiatives, error: initiativesError } = await supabase
        .from('decarbonization_initiatives')
        .select('*')
        .eq('user_id', user.id);

      if (initiativesError) throw initiativesError;

      const targetSummary = targets?.reduce((acc, target) => {
        acc.totalTargets++;
        switch (target.status) {
          case 'active': acc.activeTargets++; break;
          case 'achieved': acc.achievedTargets++; break;
          case 'at_risk': acc.atRiskTargets++; break;
        }
        acc.avgProgress += Number(target.progress_percentage) || 0;
        return acc;
      }, {
        totalTargets: 0,
        activeTargets: 0,
        achievedTargets: 0,
        atRiskTargets: 0,
        avgProgress: 0
      }) || { totalTargets: 0, activeTargets: 0, achievedTargets: 0, atRiskTargets: 0, avgProgress: 0 };

      const initiativeSummary = initiatives?.reduce((acc, initiative) => {
        acc.totalInitiatives++;
        switch (initiative.status) {
          case 'planned': acc.plannedInitiatives++; break;
          case 'in_progress': acc.inProgressInitiatives++; break;
          case 'completed': acc.completedInitiatives++; break;
        }
        acc.totalReductionPotential += Number(initiative.estimated_reduction) || 0;
        return acc;
      }, {
        totalInitiatives: 0,
        plannedInitiatives: 0,
        inProgressInitiatives: 0,
        completedInitiatives: 0,
        totalReductionPotential: 0
      }) || { totalInitiatives: 0, plannedInitiatives: 0, inProgressInitiatives: 0, completedInitiatives: 0, totalReductionPotential: 0 };

      setSummary({
        ...targetSummary,
        ...initiativeSummary,
        avgProgress: targetSummary.totalTargets > 0 ? targetSummary.avgProgress / targetSummary.totalTargets : 0
      });
    } catch (error) {
      console.error('Error fetching net-zero summary:', error);
    } finally {
      setIsLoading(false);
    }
  };

  if (isLoading) {
    return <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
      {[1, 2, 3, 4, 5, 6].map(i => (
        <Card key={i} className="animate-pulse">
          <div className="h-24 bg-gray-200 rounded"></div>
        </Card>
      ))}
    </div>;
  }

  return (
    <div className="space-y-6">
      {/* Key Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card className="bg-blue-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Target className="h-5 w-5 text-blue-600" />
              <div>
                <p className="text-sm text-gray-600">Total Targets</p>
                <p className="text-2xl font-bold">{summary.totalTargets}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-green-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <CheckCircle className="h-5 w-5 text-green-600" />
              <div>
                <p className="text-sm text-gray-600">Active Targets</p>
                <p className="text-2xl font-bold">{summary.activeTargets}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-purple-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <TrendingDown className="h-5 w-5 text-purple-600" />
              <div>
                <p className="text-sm text-gray-600">Total Initiatives</p>
                <p className="text-2xl font-bold">{summary.totalInitiatives}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-orange-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-5 w-5 text-orange-600" />
              <div>
                <p className="text-sm text-gray-600">Reduction Potential</p>
                <p className="text-2xl font-bold">{summary.totalReductionPotential.toFixed(1)}</p>
                <p className="text-sm text-gray-500">tCO2e/year</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Progress Overview */}
      <Card>
        <CardHeader>
          <CardTitle>Overall Progress</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            <div>
              <div className="flex justify-between items-center mb-2">
                <span className="text-sm font-medium">Average Target Progress</span>
                <span className="text-sm text-gray-600">{summary.avgProgress.toFixed(1)}%</span>
              </div>
              <Progress value={summary.avgProgress} className="h-2" />
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Status Breakdown */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Target Status</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              <div className="flex justify-between items-center">
                <span className="flex items-center gap-2">
                  <div className="w-3 h-3 bg-green-500 rounded-full"></div>
                  Active
                </span>
                <Badge variant="outline">{summary.activeTargets}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="flex items-center gap-2">
                  <div className="w-3 h-3 bg-blue-500 rounded-full"></div>
                  Achieved
                </span>
                <Badge variant="outline">{summary.achievedTargets}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="flex items-center gap-2">
                  <div className="w-3 h-3 bg-red-500 rounded-full"></div>
                  At Risk
                </span>
                <Badge variant="outline">{summary.atRiskTargets}</Badge>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Initiative Status</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              <div className="flex justify-between items-center">
                <span className="flex items-center gap-2">
                  <div className="w-3 h-3 bg-gray-500 rounded-full"></div>
                  Planned
                </span>
                <Badge variant="outline">{summary.plannedInitiatives}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="flex items-center gap-2">
                  <div className="w-3 h-3 bg-yellow-500 rounded-full"></div>
                  In Progress
                </span>
                <Badge variant="outline">{summary.inProgressInitiatives}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="flex items-center gap-2">
                  <div className="w-3 h-3 bg-green-500 rounded-full"></div>
                  Completed
                </span>
                <Badge variant="outline">{summary.completedInitiatives}</Badge>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};
