
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Progress } from '@/components/ui/progress';
import { Badge } from '@/components/ui/badge';
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, BarChart, Bar } from 'recharts';
import { TrendingDown, Calendar, CheckCircle, AlertTriangle } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';

interface PathwayData {
  year: number;
  targetEmissions: number;
  actualEmissions?: number;
  projectedEmissions?: number;
}

export const PathwayTracker: React.FC = () => {
  const [pathwayData, setPathwayData] = useState<PathwayData[]>([]);
  const [initiativeImpact, setInitiativeImpact] = useState<any[]>([]);
  const [milestones, setMilestones] = useState<any[]>([]);
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    fetchPathwayData();
  }, []);

  const fetchPathwayData = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      // Fetch targets
      const { data: targets, error: targetsError } = await supabase
        .from('net_zero_targets')
        .select('*')
        .eq('user_id', user.id)
        .eq('status', 'active');

      if (targetsError) throw targetsError;

      // Fetch initiatives
      const { data: initiatives, error: initiativesError } = await supabase
        .from('decarbonization_initiatives')
        .select('*')
        .eq('user_id', user.id);

      if (initiativesError) throw initiativesError;

      if (targets && targets.length > 0) {
        // Create pathway data
        const pathway: PathwayData[] = [];
        const currentYear = new Date().getFullYear();
        
        targets.forEach(target => {
          const yearRange = target.target_year - target.baseline_year;
          const reductionPerYear = (target.baseline_emissions - target.target_emissions) / yearRange;
          
          for (let year = target.baseline_year; year <= target.target_year; year++) {
            const existingData = pathway.find(p => p.year === year);
            const targetEmissions = target.baseline_emissions - (reductionPerYear * (year - target.baseline_year));
            
            if (existingData) {
              existingData.targetEmissions = Math.min(existingData.targetEmissions, targetEmissions);
            } else {
              pathway.push({
                year,
                targetEmissions,
                actualEmissions: year === currentYear ? target.current_emissions : undefined
              });
            }
          }
        });

        setPathwayData(pathway.sort((a, b) => a.year - b.year));

        // Process initiative impact
        const impactData = initiatives?.reduce((acc: any, initiative) => {
          const type = initiative.initiative_type.replace('_', ' ');
          if (!acc[type]) {
            acc[type] = { type, reduction: 0, investment: 0 };
          }
          acc[type].reduction += Number(initiative.estimated_reduction) || 0;
          acc[type].investment += Number(initiative.investment_required) || 0;
          return acc;
        }, {});

        setInitiativeImpact(Object.values(impactData || {}));

        // Create milestones
        const milestoneData = targets?.map(target => ({
          id: target.id,
          name: target.target_name,
          year: target.target_year,
          progress: target.progress_percentage || 0,
          status: target.status,
          reduction: target.reduction_percentage || 0
        })) || [];

        setMilestones(milestoneData);
      }
    } catch (error) {
      console.error('Error fetching pathway data:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'achieved': return 'bg-blue-100 text-blue-800';
      case 'at_risk': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  if (isLoading) {
    return <div className="space-y-4">
      {[1, 2, 3].map(i => (
        <Card key={i} className="animate-pulse">
          <div className="h-64 bg-gray-200 rounded"></div>
        </Card>
      ))}
    </div>;
  }

  return (
    <div className="space-y-6">
      {/* Emission Pathway Chart */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingDown className="h-5 w-5" />
            Emission Reduction Pathway
          </CardTitle>
        </CardHeader>
        <CardContent>
          <ResponsiveContainer width="100%" height={400}>
            <LineChart data={pathwayData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="year" />
              <YAxis />
              <Tooltip />
              <Line 
                type="monotone" 
                dataKey="targetEmissions" 
                stroke="#8884d8" 
                strokeWidth={2}
                name="Target Emissions (tCO2e)"
              />
              <Line 
                type="monotone" 
                dataKey="actualEmissions" 
                stroke="#82ca9d" 
                strokeWidth={2}
                name="Actual Emissions (tCO2e)"
              />
            </LineChart>
          </ResponsiveContainer>
        </CardContent>
      </Card>

      {/* Initiative Impact */}
      <Card>
        <CardHeader>
          <CardTitle>Initiative Impact by Type</CardTitle>
        </CardHeader>
        <CardContent>
          <ResponsiveContainer width="100%" height={300}>
            <BarChart data={initiativeImpact}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="type" />
              <YAxis />
              <Tooltip />
              <Bar dataKey="reduction" fill="#8884d8" name="Reduction (tCO2e/year)" />
            </BarChart>
          </ResponsiveContainer>
        </CardContent>
      </Card>

      {/* Milestones */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="h-5 w-5" />
            Target Milestones
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {milestones.map((milestone) => (
              <div key={milestone.id} className="border rounded-lg p-4">
                <div className="flex justify-between items-start mb-3">
                  <div>
                    <h4 className="font-semibold">{milestone.name}</h4>
                    <p className="text-sm text-gray-600">Target Year: {milestone.year}</p>
                  </div>
                  <div className="text-right">
                    <Badge className={getStatusColor(milestone.status)}>
                      {milestone.status}
                    </Badge>
                    <p className="text-sm text-gray-600 mt-1">
                      {milestone.reduction.toFixed(1)}% reduction
                    </p>
                  </div>
                </div>
                
                <div className="space-y-2">
                  <div className="flex justify-between items-center">
                    <span className="text-sm font-medium">Progress</span>
                    <span className="text-sm text-gray-600">{milestone.progress.toFixed(1)}%</span>
                  </div>
                  <Progress value={milestone.progress} className="h-2" />
                </div>
              </div>
            ))}
          </div>
          
          {milestones.length === 0 && (
            <div className="text-center py-8">
              <Calendar className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <p className="text-gray-500">No milestones available. Set up your targets to track progress.</p>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Key Insights */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <CheckCircle className="h-5 w-5" />
            Key Insights
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {pathwayData.length > 0 && (
              <div className="p-3 bg-blue-50 rounded-lg">
                <p className="text-sm text-blue-800">
                  <strong>Pathway Analysis:</strong> Your emission reduction pathway spans {pathwayData.length} years
                  with targets across multiple scopes.
                </p>
              </div>
            )}
            
            {initiativeImpact.length > 0 && (
              <div className="p-3 bg-green-50 rounded-lg">
                <p className="text-sm text-green-800">
                  <strong>Initiative Portfolio:</strong> You have {initiativeImpact.length} different types of 
                  decarbonization initiatives with a total reduction potential of{' '}
                  {initiativeImpact.reduce((sum, item) => sum + item.reduction, 0).toFixed(1)} tCO2e/year.
                </p>
              </div>
            )}
            
            {milestones.filter(m => m.progress >= 80).length > 0 && (
              <div className="p-3 bg-purple-50 rounded-lg">
                <p className="text-sm text-purple-800">
                  <strong>Progress Update:</strong> {milestones.filter(m => m.progress >= 80).length} of your 
                  targets are on track (≥80% progress). Keep up the good work!
                </p>
              </div>
            )}
            
            {milestones.filter(m => m.status === 'at_risk').length > 0 && (
              <div className="p-3 bg-red-50 rounded-lg">
                <p className="text-sm text-red-800">
                  <strong>Risk Alert:</strong> {milestones.filter(m => m.status === 'at_risk').length} target(s) 
                  are at risk. Consider accelerating related initiatives or setting interim targets.
                </p>
              </div>
            )}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
