
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Plus, Edit, Trash2, Target } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface NetZeroTarget {
  id: string;
  target_name: string;
  target_type: string;
  scope: string;
  baseline_year: number;
  baseline_emissions: number;
  target_year: number;
  target_emissions: number;
  reduction_percentage: number;
  current_emissions: number;
  progress_percentage: number;
  status: string;
  methodology: string;
  verification_status: string;
}

interface TargetManagerProps {
  onUpdate: () => void;
}

export const TargetManager: React.FC<TargetManagerProps> = ({ onUpdate }) => {
  const [targets, setTargets] = useState<NetZeroTarget[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [showForm, setShowForm] = useState(false);
  const [editingTarget, setEditingTarget] = useState<NetZeroTarget | null>(null);
  const [formData, setFormData] = useState({
    target_name: '',
    target_type: 'absolute',
    scope: 'scope-1',
    baseline_year: new Date().getFullYear() - 1,
    baseline_emissions: '',
    target_year: new Date().getFullYear() + 10,
    target_emissions: '',
    current_emissions: '',
    status: 'active',
    methodology: 'SBTi',
    verification_status: 'unverified'
  });
  const { toast } = useToast();

  useEffect(() => {
    fetchTargets();
  }, []);

  const fetchTargets = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from('net_zero_targets')
        .select('*')
        .eq('user_id', user.id)
        .order('created_at', { ascending: false });

      if (error) throw error;
      setTargets(data || []);
    } catch (error) {
      console.error('Error fetching targets:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('User not authenticated');

      const targetData = {
        user_id: user.id,
        target_name: formData.target_name,
        target_type: formData.target_type,
        scope: formData.scope,
        baseline_year: formData.baseline_year,
        baseline_emissions: parseFloat(formData.baseline_emissions),
        target_year: formData.target_year,
        target_emissions: parseFloat(formData.target_emissions),
        current_emissions: formData.current_emissions ? parseFloat(formData.current_emissions) : null,
        status: formData.status,
        methodology: formData.methodology,
        verification_status: formData.verification_status
      };

      if (editingTarget) {
        const { error } = await supabase
          .from('net_zero_targets')
          .update(targetData)
          .eq('id', editingTarget.id);
        
        if (error) throw error;
        toast({ title: "Target updated successfully" });
      } else {
        const { error } = await supabase
          .from('net_zero_targets')
          .insert(targetData);
        
        if (error) throw error;
        toast({ title: "Target added successfully" });
      }

      resetForm();
      fetchTargets();
      onUpdate();
    } catch (error) {
      console.error('Error saving target:', error);
      toast({
        title: "Error saving target",
        description: "Please try again",
        variant: "destructive"
      });
    }
  };

  const resetForm = () => {
    setFormData({
      target_name: '',
      target_type: 'absolute',
      scope: 'scope-1',
      baseline_year: new Date().getFullYear() - 1,
      baseline_emissions: '',
      target_year: new Date().getFullYear() + 10,
      target_emissions: '',
      current_emissions: '',
      status: 'active',
      methodology: 'SBTi',
      verification_status: 'unverified'
    });
    setEditingTarget(null);
    setShowForm(false);
  };

  const handleEdit = (target: NetZeroTarget) => {
    setFormData({
      target_name: target.target_name,
      target_type: target.target_type,
      scope: target.scope,
      baseline_year: target.baseline_year,
      baseline_emissions: target.baseline_emissions.toString(),
      target_year: target.target_year,
      target_emissions: target.target_emissions.toString(),
      current_emissions: target.current_emissions?.toString() || '',
      status: target.status,
      methodology: target.methodology || 'SBTi',
      verification_status: target.verification_status
    });
    setEditingTarget(target);
    setShowForm(true);
  };

  const handleDelete = async (id: string) => {
    try {
      const { error } = await supabase
        .from('net_zero_targets')
        .delete()
        .eq('id', id);

      if (error) throw error;
      
      toast({ title: "Target deleted successfully" });
      fetchTargets();
      onUpdate();
    } catch (error) {
      console.error('Error deleting target:', error);
      toast({
        title: "Error deleting target",
        variant: "destructive"
      });
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'achieved': return 'bg-blue-100 text-blue-800';
      case 'at_risk': return 'bg-red-100 text-red-800';
      case 'failed': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getVerificationColor = (status: string) => {
    switch (status) {
      case 'verified': return 'bg-green-100 text-green-800';
      case 'pending': return 'bg-yellow-100 text-yellow-800';
      case 'unverified': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  if (isLoading) {
    return <div className="animate-pulse h-64 bg-gray-200 rounded"></div>;
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold">Net-Zero Targets</h3>
        <Button onClick={() => setShowForm(true)} className="flex items-center gap-2">
          <Plus className="h-4 w-4" />
          Add Target
        </Button>
      </div>

      {showForm && (
        <Card>
          <CardHeader>
            <CardTitle>
              {editingTarget ? 'Edit Target' : 'Add New Target'}
            </CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <Label>Target Name *</Label>
                  <Input
                    value={formData.target_name}
                    onChange={(e) => setFormData({...formData, target_name: e.target.value})}
                    required
                  />
                </div>
                <div>
                  <Label>Target Type *</Label>
                  <Select value={formData.target_type} onValueChange={(value) => setFormData({...formData, target_type: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="absolute">Absolute</SelectItem>
                      <SelectItem value="intensity">Intensity</SelectItem>
                      <SelectItem value="renewable_energy">Renewable Energy</SelectItem>
                      <SelectItem value="carbon_neutral">Carbon Neutral</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Scope *</Label>
                  <Select value={formData.scope} onValueChange={(value) => setFormData({...formData, scope: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="scope-1">Scope 1</SelectItem>
                      <SelectItem value="scope-2">Scope 2</SelectItem>
                      <SelectItem value="scope-3">Scope 3</SelectItem>
                      <SelectItem value="all-scopes">All Scopes</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Methodology</Label>
                  <Select value={formData.methodology} onValueChange={(value) => setFormData({...formData, methodology: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="SBTi">Science Based Targets initiative (SBTi)</SelectItem>
                      <SelectItem value="Paris_Agreement">Paris Agreement</SelectItem>
                      <SelectItem value="Custom">Custom</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Baseline Year *</Label>
                  <Input
                    type="number"
                    value={formData.baseline_year}
                    onChange={(e) => setFormData({...formData, baseline_year: parseInt(e.target.value)})}
                    required
                  />
                </div>
                <div>
                  <Label>Baseline Emissions (tCO2e) *</Label>
                  <Input
                    type="number"
                    step="0.01"
                    value={formData.baseline_emissions}
                    onChange={(e) => setFormData({...formData, baseline_emissions: e.target.value})}
                    required
                  />
                </div>
                <div>
                  <Label>Target Year *</Label>
                  <Input
                    type="number"
                    value={formData.target_year}
                    onChange={(e) => setFormData({...formData, target_year: parseInt(e.target.value)})}
                    required
                  />
                </div>
                <div>
                  <Label>Target Emissions (tCO2e) *</Label>
                  <Input
                    type="number"
                    step="0.01"
                    value={formData.target_emissions}
                    onChange={(e) => setFormData({...formData, target_emissions: e.target.value})}
                    required
                  />
                </div>
                <div>
                  <Label>Current Emissions (tCO2e)</Label>
                  <Input
                    type="number"
                    step="0.01"
                    value={formData.current_emissions}
                    onChange={(e) => setFormData({...formData, current_emissions: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Status</Label>
                  <Select value={formData.status} onValueChange={(value) => setFormData({...formData, status: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="active">Active</SelectItem>
                      <SelectItem value="achieved">Achieved</SelectItem>
                      <SelectItem value="at_risk">At Risk</SelectItem>
                      <SelectItem value="failed">Failed</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="flex gap-2">
                <Button type="submit">
                  {editingTarget ? 'Update Target' : 'Add Target'}
                </Button>
                <Button type="button" variant="outline" onClick={resetForm}>
                  Cancel
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>
      )}

      <div className="grid grid-cols-1 gap-4">
        {targets.map((target) => (
          <Card key={target.id}>
            <CardContent className="p-4">
              <div className="flex justify-between items-start mb-4">
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-2">
                    <h4 className="font-semibold">{target.target_name}</h4>
                    <Badge className={getStatusColor(target.status)}>
                      {target.status}
                    </Badge>
                    <Badge className={getVerificationColor(target.verification_status)}>
                      {target.verification_status}
                    </Badge>
                  </div>
                  
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm mb-4">
                    <div>
                      <p className="text-gray-600">Type</p>
                      <p className="font-medium">{target.target_type}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Scope</p>
                      <p className="font-medium">{target.scope}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Baseline</p>
                      <p className="font-medium">{target.baseline_emissions.toLocaleString()} tCO2e ({target.baseline_year})</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Target</p>
                      <p className="font-medium">{target.target_emissions.toLocaleString()} tCO2e ({target.target_year})</p>
                    </div>
                  </div>

                  <div className="space-y-2">
                    <div className="flex justify-between items-center">
                      <span className="text-sm font-medium">Progress</span>
                      <span className="text-sm text-gray-600">{target.progress_percentage?.toFixed(1) || 0}%</span>
                    </div>
                    <Progress value={target.progress_percentage || 0} className="h-2" />
                    <div className="flex justify-between text-xs text-gray-500">
                      <span>Reduction Target: {target.reduction_percentage?.toFixed(1) || 0}%</span>
                      {target.current_emissions && (
                        <span>Current: {target.current_emissions.toLocaleString()} tCO2e</span>
                      )}
                    </div>
                  </div>
                </div>
                
                <div className="flex gap-2">
                  <Button size="sm" variant="outline" onClick={() => handleEdit(target)}>
                    <Edit className="h-4 w-4" />
                  </Button>
                  <Button size="sm" variant="outline" onClick={() => handleDelete(target.id)}>
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {targets.length === 0 && (
        <Card>
          <CardContent className="p-8 text-center">
            <Target className="h-12 w-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-600 mb-2">No targets set yet</h3>
            <p className="text-gray-500 mb-4">Start your net-zero journey by setting your first emission reduction target.</p>
            <Button onClick={() => setShowForm(true)}>
              Set First Target
            </Button>
          </CardContent>
        </Card>
      )}
    </div>
  );
};
