
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart, Pie, Cell } from 'recharts';
import { supabase } from '@/integrations/supabase/client';

export const ESGAnalytics: React.FC = () => {
  const [sectorData, setSectorData] = useState<any[]>([]);
  const [esgBreakdown, setEsgBreakdown] = useState<any[]>([]);
  const [investmentTypes, setInvestmentTypes] = useState<any[]>([]);
  const [isLoading, setIsLoading] = useState(true);

  const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042', '#8884D8'];

  useEffect(() => {
    fetchAnalyticsData();
  }, []);

  const fetchAnalyticsData = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data: investments, error } = await supabase
        .from('portfolio_investments')
        .select('*')
        .eq('user_id', user.id)
        .eq('status', 'active');

      if (error) throw error;

      if (investments && investments.length > 0) {
        // Sector analysis
        const sectorAnalysis = investments.reduce((acc: any, inv) => {
          const sector = inv.sector || 'Unknown';
          if (!acc[sector]) {
            acc[sector] = {
              sector,
              totalValue: 0,
              avgESGScore: 0,
              count: 0,
              esgSum: 0
            };
          }
          acc[sector].totalValue += Number(inv.investment_amount);
          acc[sector].esgSum += Number(inv.esg_score) || 0;
          acc[sector].count += 1;
          acc[sector].avgESGScore = acc[sector].esgSum / acc[sector].count;
          return acc;
        }, {});

        setSectorData(Object.values(sectorAnalysis));

        // ESG breakdown
        const avgScores = investments.reduce((acc: any, inv) => {
          acc.environmental += Number(inv.environmental_score) || 0;
          acc.social += Number(inv.social_score) || 0;
          acc.governance += Number(inv.governance_score) || 0;
          return acc;
        }, { environmental: 0, social: 0, governance: 0 });

        const count = investments.length;
        setEsgBreakdown([
          { name: 'Environmental', value: (avgScores.environmental / count).toFixed(1) },
          { name: 'Social', value: (avgScores.social / count).toFixed(1) },
          { name: 'Governance', value: (avgScores.governance / count).toFixed(1) }
        ]);

        // Investment types
        const typeAnalysis = investments.reduce((acc: any, inv) => {
          const type = inv.investment_type;
          if (!acc[type]) {
            acc[type] = { name: type, value: 0, count: 0 };
          }
          acc[type].value += Number(inv.investment_amount);
          acc[type].count += 1;
          return acc;
        }, {});

        setInvestmentTypes(Object.values(typeAnalysis));
      }
    } catch (error) {
      console.error('Error fetching analytics data:', error);
    } finally {
      setIsLoading(false);
    }
  };

  if (isLoading) {
    return <div className="space-y-4">
      {[1, 2, 3].map(i => (
        <Card key={i} className="animate-pulse">
          <div className="h-64 bg-gray-200 rounded"></div>
        </Card>
      ))}
    </div>;
  }

  return (
    <div className="space-y-6">
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>ESG Scores by Sector</CardTitle>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={sectorData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="sector" />
                <YAxis />
                <Tooltip />
                <Bar dataKey="avgESGScore" fill="#8884d8" />
              </BarChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>ESG Component Breakdown</CardTitle>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={esgBreakdown}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="name" />
                <YAxis />
                <Tooltip />
                <Bar dataKey="value" fill="#82ca9d" />
              </BarChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Investment Allocation by Type</CardTitle>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={investmentTypes}
                  cx="50%"
                  cy="50%"
                  labelLine={false}
                  label={({ name, percent }) => `${name} ${(percent * 100).toFixed(0)}%`}
                  outerRadius={80}
                  fill="#8884d8"
                  dataKey="value"
                >
                  {investmentTypes.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                  ))}
                </Pie>
                <Tooltip />
              </PieChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Sector Value Distribution</CardTitle>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={sectorData}
                  cx="50%"
                  cy="50%"
                  labelLine={false}
                  label={({ sector, percent }) => `${sector} ${(percent * 100).toFixed(0)}%`}
                  outerRadius={80}
                  fill="#8884d8"
                  dataKey="totalValue"
                >
                  {sectorData.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                  ))}
                </Pie>
                <Tooltip />
              </PieChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};
