
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Plus, Edit, Trash2, TrendingUp } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface Investment {
  id: string;
  investment_name: string;
  investment_type: string;
  sector: string;
  region: string;
  investment_amount: number;
  currency: string;
  esg_score: number;
  environmental_score: number;
  social_score: number;
  governance_score: number;
  carbon_intensity: number;
  investment_date: string;
  status: string;
  notes: string;
}

interface InvestmentManagerProps {
  onUpdate: () => void;
}

export const InvestmentManager: React.FC<InvestmentManagerProps> = ({ onUpdate }) => {
  const [investments, setInvestments] = useState<Investment[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [showForm, setShowForm] = useState(false);
  const [editingInvestment, setEditingInvestment] = useState<Investment | null>(null);
  const [formData, setFormData] = useState({
    investment_name: '',
    investment_type: 'equity',
    sector: '',
    region: '',
    investment_amount: '',
    currency: 'USD',
    esg_score: '',
    environmental_score: '',
    social_score: '',
    governance_score: '',
    carbon_intensity: '',
    investment_date: '',
    status: 'active',
    notes: ''
  });
  const { toast } = useToast();

  useEffect(() => {
    fetchInvestments();
  }, []);

  const fetchInvestments = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from('portfolio_investments')
        .select('*')
        .eq('user_id', user.id)
        .order('created_at', { ascending: false });

      if (error) throw error;
      setInvestments(data || []);
    } catch (error) {
      console.error('Error fetching investments:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('User not authenticated');

      const investmentData = {
        user_id: user.id,
        investment_name: formData.investment_name,
        investment_type: formData.investment_type,
        sector: formData.sector,
        region: formData.region,
        investment_amount: parseFloat(formData.investment_amount),
        currency: formData.currency,
        esg_score: formData.esg_score ? parseFloat(formData.esg_score) : null,
        environmental_score: formData.environmental_score ? parseFloat(formData.environmental_score) : null,
        social_score: formData.social_score ? parseFloat(formData.social_score) : null,
        governance_score: formData.governance_score ? parseFloat(formData.governance_score) : null,
        carbon_intensity: formData.carbon_intensity ? parseFloat(formData.carbon_intensity) : null,
        investment_date: formData.investment_date,
        status: formData.status,
        notes: formData.notes
      };

      if (editingInvestment) {
        const { error } = await supabase
          .from('portfolio_investments')
          .update(investmentData)
          .eq('id', editingInvestment.id);
        
        if (error) throw error;
        toast({ title: "Investment updated successfully" });
      } else {
        const { error } = await supabase
          .from('portfolio_investments')
          .insert(investmentData);
        
        if (error) throw error;
        toast({ title: "Investment added successfully" });
      }

      resetForm();
      fetchInvestments();
      onUpdate();
    } catch (error) {
      console.error('Error saving investment:', error);
      toast({
        title: "Error saving investment",
        description: "Please try again",
        variant: "destructive"
      });
    }
  };

  const resetForm = () => {
    setFormData({
      investment_name: '',
      investment_type: 'equity',
      sector: '',
      region: '',
      investment_amount: '',
      currency: 'USD',
      esg_score: '',
      environmental_score: '',
      social_score: '',
      governance_score: '',
      carbon_intensity: '',
      investment_date: '',
      status: 'active',
      notes: ''
    });
    setEditingInvestment(null);
    setShowForm(false);
  };

  const handleEdit = (investment: Investment) => {
    setFormData({
      investment_name: investment.investment_name,
      investment_type: investment.investment_type,
      sector: investment.sector || '',
      region: investment.region || '',
      investment_amount: investment.investment_amount.toString(),
      currency: investment.currency,
      esg_score: investment.esg_score?.toString() || '',
      environmental_score: investment.environmental_score?.toString() || '',
      social_score: investment.social_score?.toString() || '',
      governance_score: investment.governance_score?.toString() || '',
      carbon_intensity: investment.carbon_intensity?.toString() || '',
      investment_date: investment.investment_date,
      status: investment.status,
      notes: investment.notes || ''
    });
    setEditingInvestment(investment);
    setShowForm(true);
  };

  const handleDelete = async (id: string) => {
    try {
      const { error } = await supabase
        .from('portfolio_investments')
        .delete()
        .eq('id', id);

      if (error) throw error;
      
      toast({ title: "Investment deleted successfully" });
      fetchInvestments();
      onUpdate();
    } catch (error) {
      console.error('Error deleting investment:', error);
      toast({
        title: "Error deleting investment",
        variant: "destructive"
      });
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'divested': return 'bg-gray-100 text-gray-800';
      case 'under_review': return 'bg-yellow-100 text-yellow-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getESGScoreColor = (score: number) => {
    if (score >= 80) return 'text-green-600';
    if (score >= 60) return 'text-yellow-600';
    return 'text-red-600';
  };

  if (isLoading) {
    return <div className="animate-pulse h-64 bg-gray-200 rounded"></div>;
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold">Investment Portfolio</h3>
        <Button onClick={() => setShowForm(true)} className="flex items-center gap-2">
          <Plus className="h-4 w-4" />
          Add Investment
        </Button>
      </div>

      {showForm && (
        <Card>
          <CardHeader>
            <CardTitle>
              {editingInvestment ? 'Edit Investment' : 'Add New Investment'}
            </CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <Label>Investment Name *</Label>
                  <Input
                    value={formData.investment_name}
                    onChange={(e) => setFormData({...formData, investment_name: e.target.value})}
                    required
                  />
                </div>
                <div>
                  <Label>Investment Type *</Label>
                  <Select value={formData.investment_type} onValueChange={(value) => setFormData({...formData, investment_type: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="equity">Equity</SelectItem>
                      <SelectItem value="bond">Bond</SelectItem>
                      <SelectItem value="fund">Fund</SelectItem>
                      <SelectItem value="real_estate">Real Estate</SelectItem>
                      <SelectItem value="alternative">Alternative</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Sector</Label>
                  <Input
                    value={formData.sector}
                    onChange={(e) => setFormData({...formData, sector: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Region</Label>
                  <Input
                    value={formData.region}
                    onChange={(e) => setFormData({...formData, region: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Investment Amount *</Label>
                  <Input
                    type="number"
                    value={formData.investment_amount}
                    onChange={(e) => setFormData({...formData, investment_amount: e.target.value})}
                    required
                  />
                </div>
                <div>
                  <Label>Currency</Label>
                  <Select value={formData.currency} onValueChange={(value) => setFormData({...formData, currency: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="USD">USD</SelectItem>
                      <SelectItem value="EUR">EUR</SelectItem>
                      <SelectItem value="GBP">GBP</SelectItem>
                      <SelectItem value="JPY">JPY</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>ESG Score (0-100)</Label>
                  <Input
                    type="number"
                    min="0"
                    max="100"
                    value={formData.esg_score}
                    onChange={(e) => setFormData({...formData, esg_score: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Environmental Score (0-100)</Label>
                  <Input
                    type="number"
                    min="0"
                    max="100"
                    value={formData.environmental_score}
                    onChange={(e) => setFormData({...formData, environmental_score: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Social Score (0-100)</Label>
                  <Input
                    type="number"
                    min="0"
                    max="100"
                    value={formData.social_score}
                    onChange={(e) => setFormData({...formData, social_score: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Governance Score (0-100)</Label>
                  <Input
                    type="number"
                    min="0"
                    max="100"
                    value={formData.governance_score}
                    onChange={(e) => setFormData({...formData, governance_score: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Carbon Intensity</Label>
                  <Input
                    type="number"
                    value={formData.carbon_intensity}
                    onChange={(e) => setFormData({...formData, carbon_intensity: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Investment Date *</Label>
                  <Input
                    type="date"
                    value={formData.investment_date}
                    onChange={(e) => setFormData({...formData, investment_date: e.target.value})}
                    required
                  />
                </div>
              </div>
              
              <div>
                <Label>Status</Label>
                <Select value={formData.status} onValueChange={(value) => setFormData({...formData, status: value})}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="active">Active</SelectItem>
                    <SelectItem value="divested">Divested</SelectItem>
                    <SelectItem value="under_review">Under Review</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div>
                <Label>Notes</Label>
                <Input
                  value={formData.notes}
                  onChange={(e) => setFormData({...formData, notes: e.target.value})}
                  placeholder="Additional notes about this investment"
                />
              </div>

              <div className="flex gap-2">
                <Button type="submit">
                  {editingInvestment ? 'Update Investment' : 'Add Investment'}
                </Button>
                <Button type="button" variant="outline" onClick={resetForm}>
                  Cancel
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>
      )}

      <div className="grid grid-cols-1 gap-4">
        {investments.map((investment) => (
          <Card key={investment.id}>
            <CardContent className="p-4">
              <div className="flex justify-between items-start">
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-2">
                    <h4 className="font-semibold">{investment.investment_name}</h4>
                    <Badge className={getStatusColor(investment.status)}>
                      {investment.status}
                    </Badge>
                  </div>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                    <div>
                      <p className="text-gray-600">Type</p>
                      <p className="font-medium">{investment.investment_type}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Amount</p>
                      <p className="font-medium">{investment.investment_amount.toLocaleString()} {investment.currency}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">ESG Score</p>
                      <p className={`font-medium ${getESGScoreColor(investment.esg_score || 0)}`}>
                        {investment.esg_score ? investment.esg_score.toFixed(1) : 'N/A'}
                      </p>
                    </div>
                    <div>
                      <p className="text-gray-600">Date</p>
                      <p className="font-medium">{new Date(investment.investment_date).toLocaleDateString()}</p>
                    </div>
                  </div>
                  {investment.sector && (
                    <div className="mt-2">
                      <Badge variant="outline">{investment.sector}</Badge>
                      {investment.region && <Badge variant="outline" className="ml-2">{investment.region}</Badge>}
                    </div>
                  )}
                </div>
                <div className="flex gap-2">
                  <Button size="sm" variant="outline" onClick={() => handleEdit(investment)}>
                    <Edit className="h-4 w-4" />
                  </Button>
                  <Button size="sm" variant="outline" onClick={() => handleDelete(investment.id)}>
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {investments.length === 0 && (
        <Card>
          <CardContent className="p-8 text-center">
            <TrendingUp className="h-12 w-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-600 mb-2">No investments yet</h3>
            <p className="text-gray-500 mb-4">Start building your ESG portfolio by adding your first investment.</p>
            <Button onClick={() => setShowForm(true)}>
              Add First Investment
            </Button>
          </CardContent>
        </Card>
      )}
    </div>
  );
};
