
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { TrendingUp, TrendingDown, DollarSign, Target, Leaf, Users } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';

interface PortfolioSummary {
  totalValue: number;
  totalInvestments: number;
  avgESGScore: number;
  avgEnvironmentalScore: number;
  avgSocialScore: number;
  avgGovernanceScore: number;
  currency: string;
  carbonIntensity: number;
}

export const PortfolioOverview: React.FC = () => {
  const [summary, setSummary] = useState<PortfolioSummary>({
    totalValue: 0,
    totalInvestments: 0,
    avgESGScore: 0,
    avgEnvironmentalScore: 0,
    avgSocialScore: 0,
    avgGovernanceScore: 0,
    currency: 'USD',
    carbonIntensity: 0
  });
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    fetchPortfolioSummary();
  }, []);

  const fetchPortfolioSummary = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data: investments, error } = await supabase
        .from('portfolio_investments')
        .select('*')
        .eq('user_id', user.id)
        .eq('status', 'active');

      if (error) throw error;

      if (investments && investments.length > 0) {
        const totalValue = investments.reduce((sum, inv) => sum + Number(inv.investment_amount), 0);
        const avgESGScore = investments.reduce((sum, inv) => sum + (Number(inv.esg_score) || 0), 0) / investments.length;
        const avgEnvironmentalScore = investments.reduce((sum, inv) => sum + (Number(inv.environmental_score) || 0), 0) / investments.length;
        const avgSocialScore = investments.reduce((sum, inv) => sum + (Number(inv.social_score) || 0), 0) / investments.length;
        const avgGovernanceScore = investments.reduce((sum, inv) => sum + (Number(inv.governance_score) || 0), 0) / investments.length;
        const avgCarbonIntensity = investments.reduce((sum, inv) => sum + (Number(inv.carbon_intensity) || 0), 0) / investments.length;

        setSummary({
          totalValue,
          totalInvestments: investments.length,
          avgESGScore,
          avgEnvironmentalScore,
          avgSocialScore,
          avgGovernanceScore,
          currency: investments[0]?.currency || 'USD',
          carbonIntensity: avgCarbonIntensity
        });
      }
    } catch (error) {
      console.error('Error fetching portfolio summary:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const getScoreColor = (score: number) => {
    if (score >= 80) return 'text-green-600 bg-green-50';
    if (score >= 60) return 'text-yellow-600 bg-yellow-50';
    return 'text-red-600 bg-red-50';
  };

  const getScoreIcon = (score: number) => {
    return score >= 70 ? <TrendingUp className="h-4 w-4" /> : <TrendingDown className="h-4 w-4" />;
  };

  if (isLoading) {
    return <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
      {[1, 2, 3, 4, 5, 6].map(i => (
        <Card key={i} className="animate-pulse">
          <div className="h-24 bg-gray-200 rounded"></div>
        </Card>
      ))}
    </div>;
  }

  return (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card className="bg-blue-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <DollarSign className="h-5 w-5 text-blue-600" />
              <div>
                <p className="text-sm text-gray-600">Total Portfolio Value</p>
                <p className="text-2xl font-bold">{summary.totalValue.toLocaleString()}</p>
                <p className="text-sm text-gray-500">{summary.currency}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-green-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Target className="h-5 w-5 text-green-600" />
              <div>
                <p className="text-sm text-gray-600">Total Investments</p>
                <p className="text-2xl font-bold">{summary.totalInvestments}</p>
                <p className="text-sm text-gray-500">Active positions</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-purple-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Leaf className="h-5 w-5 text-purple-600" />
              <div>
                <p className="text-sm text-gray-600">Carbon Intensity</p>
                <p className="text-2xl font-bold">{summary.carbonIntensity.toFixed(1)}</p>
                <p className="text-sm text-gray-500">tCO2e/M revenue</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium">Overall ESG Score</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center justify-between">
              <span className="text-2xl font-bold">{summary.avgESGScore.toFixed(1)}</span>
              <Badge className={getScoreColor(summary.avgESGScore)}>
                {getScoreIcon(summary.avgESGScore)}
              </Badge>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium flex items-center gap-1">
              <Leaf className="h-4 w-4" />
              Environmental
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center justify-between">
              <span className="text-2xl font-bold">{summary.avgEnvironmentalScore.toFixed(1)}</span>
              <Badge className={getScoreColor(summary.avgEnvironmentalScore)}>
                {getScoreIcon(summary.avgEnvironmentalScore)}
              </Badge>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium flex items-center gap-1">
              <Users className="h-4 w-4" />
              Social
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center justify-between">
              <span className="text-2xl font-bold">{summary.avgSocialScore.toFixed(1)}</span>
              <Badge className={getScoreColor(summary.avgSocialScore)}>
                {getScoreIcon(summary.avgSocialScore)}
              </Badge>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium flex items-center gap-1">
              <Target className="h-4 w-4" />
              Governance
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex items-center justify-between">
              <span className="text-2xl font-bold">{summary.avgGovernanceScore.toFixed(1)}</span>
              <Badge className={getScoreColor(summary.avgGovernanceScore)}>
                {getScoreIcon(summary.avgGovernanceScore)}
              </Badge>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
};
