
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { FileText, Download, TrendingUp, TrendingDown, AlertCircle } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface PortfolioReport {
  totalInvestments: number;
  totalValue: number;
  avgESGScore: number;
  highPerformers: number;
  lowPerformers: number;
  riskAlerts: number;
  topSectors: string[];
  currency: string;
}

export const PortfolioReporting: React.FC = () => {
  const [reportData, setReportData] = useState<PortfolioReport>({
    totalInvestments: 0,
    totalValue: 0,
    avgESGScore: 0,
    highPerformers: 0,
    lowPerformers: 0,
    riskAlerts: 0,
    topSectors: [],
    currency: 'USD'
  });
  const [isLoading, setIsLoading] = useState(true);
  const { toast } = useToast();

  useEffect(() => {
    generateReport();
  }, []);

  const generateReport = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data: investments, error } = await supabase
        .from('portfolio_investments')
        .select('*')
        .eq('user_id', user.id)
        .eq('status', 'active');

      if (error) throw error;

      if (investments && investments.length > 0) {
        const totalValue = investments.reduce((sum, inv) => sum + Number(inv.investment_amount), 0);
        const avgESGScore = investments.reduce((sum, inv) => sum + (Number(inv.esg_score) || 0), 0) / investments.length;
        
        const highPerformers = investments.filter(inv => (Number(inv.esg_score) || 0) >= 80).length;
        const lowPerformers = investments.filter(inv => (Number(inv.esg_score) || 0) < 60).length;
        const riskAlerts = investments.filter(inv => 
          (Number(inv.carbon_intensity) || 0) > 500 || (Number(inv.esg_score) || 0) < 50
        ).length;

        const sectorCounts = investments.reduce((acc: any, inv) => {
          const sector = inv.sector || 'Unknown';
          acc[sector] = (acc[sector] || 0) + 1;
          return acc;
        }, {});

        const topSectors = Object.entries(sectorCounts)
          .sort(([,a], [,b]) => (b as number) - (a as number))
          .slice(0, 3)
          .map(([sector]) => sector);

        setReportData({
          totalInvestments: investments.length,
          totalValue,
          avgESGScore,
          highPerformers,
          lowPerformers,
          riskAlerts,
          topSectors,
          currency: investments[0]?.currency || 'USD'
        });
      }
    } catch (error) {
      console.error('Error generating report:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const exportReport = () => {
    const reportContent = `
Portfolio ESG Report
Generated: ${new Date().toLocaleDateString()}

Portfolio Summary:
- Total Investments: ${reportData.totalInvestments}
- Total Value: ${reportData.totalValue.toLocaleString()} ${reportData.currency}
- Average ESG Score: ${reportData.avgESGScore.toFixed(1)}

Performance Analysis:
- High Performers (ESG ≥ 80): ${reportData.highPerformers}
- Low Performers (ESG < 60): ${reportData.lowPerformers}
- Risk Alerts: ${reportData.riskAlerts}

Top Sectors:
${reportData.topSectors.map((sector, index) => `${index + 1}. ${sector}`).join('\n')}

This report provides an overview of your portfolio's ESG performance and sustainability metrics.
    `;

    const blob = new Blob([reportContent], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `portfolio-esg-report-${new Date().toISOString().split('T')[0]}.txt`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);

    toast({
      title: "Report exported successfully",
      description: "Your portfolio ESG report has been downloaded"
    });
  };

  if (isLoading) {
    return <div className="animate-pulse h-64 bg-gray-200 rounded"></div>;
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h3 className="text-lg font-semibold">Portfolio ESG Report</h3>
          <p className="text-sm text-gray-600">Generated on {new Date().toLocaleDateString()}</p>
        </div>
        <Button onClick={exportReport} className="flex items-center gap-2">
          <Download className="h-4 w-4" />
          Export Report
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <FileText className="h-5 w-5 text-blue-600" />
              <div>
                <p className="text-sm text-gray-600">Total Investments</p>
                <p className="text-2xl font-bold">{reportData.totalInvestments}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <TrendingUp className="h-5 w-5 text-green-600" />
              <div>
                <p className="text-sm text-gray-600">Portfolio Value</p>
                <p className="text-2xl font-bold">{reportData.totalValue.toLocaleString()}</p>
                <p className="text-sm text-gray-500">{reportData.currency}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <div className={`h-5 w-5 rounded-full ${
                reportData.avgESGScore >= 80 ? 'bg-green-500' :
                reportData.avgESGScore >= 60 ? 'bg-yellow-500' : 'bg-red-500'
              }`} />
              <div>
                <p className="text-sm text-gray-600">Avg ESG Score</p>
                <p className="text-2xl font-bold">{reportData.avgESGScore.toFixed(1)}</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-green-600">
              <TrendingUp className="h-5 w-5" />
              High Performers
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-center">
              <p className="text-3xl font-bold text-green-600">{reportData.highPerformers}</p>
              <p className="text-sm text-gray-600">ESG Score ≥ 80</p>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-red-600">
              <TrendingDown className="h-5 w-5" />
              Low Performers
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-center">
              <p className="text-3xl font-bold text-red-600">{reportData.lowPerformers}</p>
              <p className="text-sm text-gray-600">ESG Score &lt; 60</p>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-orange-600">
              <AlertCircle className="h-5 w-5" />
              Risk Alerts
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-center">
              <p className="text-3xl font-bold text-orange-600">{reportData.riskAlerts}</p>
              <p className="text-sm text-gray-600">Require attention</p>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Top Sectors</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex flex-wrap gap-2">
            {reportData.topSectors.map((sector, index) => (
              <Badge key={sector} variant="outline" className="text-sm">
                #{index + 1} {sector}
              </Badge>
            ))}
          </div>
          {reportData.topSectors.length === 0 && (
            <p className="text-gray-500 text-center py-4">No sector data available</p>
          )}
        </CardContent>
      </Card>

      <Card>
        <CardHeader>
          <CardTitle>Recommendations</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {reportData.lowPerformers > 0 && (
              <div className="p-3 bg-red-50 rounded-lg">
                <p className="text-sm text-red-800">
                  <strong>Action Required:</strong> {reportData.lowPerformers} investments have ESG scores below 60. 
                  Consider reviewing these positions for improvement opportunities.
                </p>
              </div>
            )}
            {reportData.riskAlerts > 0 && (
              <div className="p-3 bg-orange-50 rounded-lg">
                <p className="text-sm text-orange-800">
                  <strong>Risk Alert:</strong> {reportData.riskAlerts} investments show elevated risk indicators. 
                  Monitor carbon intensity and ESG compliance closely.
                </p>
              </div>
            )}
            {reportData.highPerformers > reportData.totalInvestments / 2 && (
              <div className="p-3 bg-green-50 rounded-lg">
                <p className="text-sm text-green-800">
                  <strong>Well Done:</strong> Your portfolio shows strong ESG performance with {reportData.highPerformers} high-performing investments.
                </p>
              </div>
            )}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
