
import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { Calculator, Zap, Car, Building, Recycle, Target } from 'lucide-react';

interface EmissionCalculation {
  activity: string;
  amount: number;
  unit: string;
  emissionFactor: number;
  co2Equivalent: number;
  source: string;
}

export const CarbonAccounting = () => {
  const [scope1Calculations, setScope1Calculations] = useState<EmissionCalculation[]>([]);
  const [scope2Calculations, setScope2Calculations] = useState<EmissionCalculation[]>([]);
  const [scope3Calculations, setScope3Calculations] = useState<EmissionCalculation[]>([]);

  // Emission Factors (example values)
  const emissionFactors = {
    // Scope 1 - Direct emissions
    naturalGas: { factor: 0.202, unit: 'kgCO2e/kWh', source: 'EPA' },
    diesel: { factor: 2.68, unit: 'kgCO2e/liter', source: 'EPA' },
    gasoline: { factor: 2.31, unit: 'kgCO2e/liter', source: 'EPA' },
    propane: { factor: 1.51, unit: 'kgCO2e/liter', source: 'EPA' },
    
    // Scope 2 - Indirect emissions from purchased energy
    electricity: { factor: 0.855, unit: 'lbCO2e/kWh', source: 'EPA eGRID' },
    steam: { factor: 0.2, unit: 'kgCO2e/kWh', source: 'EPA' },
    
    // Scope 3 - Other indirect emissions
    businessTravel: { factor: 0.21, unit: 'kgCO2e/km', source: 'DEFRA' },
    employeeCommuting: { factor: 0.17, unit: 'kgCO2e/km', source: 'DEFRA' },
    wasteDisposal: { factor: 0.57, unit: 'tCO2e/tonne', source: 'EPA' },
    waterConsumption: { factor: 0.344, unit: 'kgCO2e/m³', source: 'EPA' }
  };

  const addCalculation = (scope: string, activity: string, amount: number, unit: string, factorKey: string) => {
    const factor = emissionFactors[factorKey as keyof typeof emissionFactors];
    if (!factor) return;

    const calculation: EmissionCalculation = {
      activity,
      amount,
      unit,
      emissionFactor: factor.factor,
      co2Equivalent: amount * factor.factor,
      source: factor.source
    };

    switch (scope) {
      case 'scope1':
        setScope1Calculations(prev => [...prev, calculation]);
        break;
      case 'scope2':
        setScope2Calculations(prev => [...prev, calculation]);
        break;
      case 'scope3':
        setScope3Calculations(prev => [...prev, calculation]);
        break;
    }
  };

  const getTotalEmissions = (calculations: EmissionCalculation[]) => {
    return calculations.reduce((total, calc) => total + calc.co2Equivalent, 0);
  };

  const getAllEmissions = () => {
    return getTotalEmissions(scope1Calculations) + 
           getTotalEmissions(scope2Calculations) + 
           getTotalEmissions(scope3Calculations);
  };

  const Scope1Calculator = () => {
    const [activity, setActivity] = useState('');
    const [amount, setAmount] = useState('');
    const [fuelType, setFuelType] = useState('naturalGas');

    const handleCalculate = () => {
      if (activity && amount) {
        addCalculation('scope1', activity, parseFloat(amount), 
          emissionFactors[fuelType as keyof typeof emissionFactors].unit, fuelType);
        setActivity('');
        setAmount('');
      }
    };

    return (
      <div className="space-y-4">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div>
            <Label>Activity Description</Label>
            <Input 
              value={activity}
              onChange={(e) => setActivity(e.target.value)}
              placeholder="e.g., Office heating"
            />
          </div>
          <div>
            <Label>Fuel Type</Label>
            <Select value={fuelType} onValueChange={setFuelType}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="naturalGas">Natural Gas</SelectItem>
                <SelectItem value="diesel">Diesel</SelectItem>
                <SelectItem value="gasoline">Gasoline</SelectItem>
                <SelectItem value="propane">Propane</SelectItem>
              </SelectContent>
            </Select>
          </div>
          <div>
            <Label>Amount</Label>
            <Input 
              type="number"
              value={amount}
              onChange={(e) => setAmount(e.target.value)}
              placeholder="Enter quantity"
            />
          </div>
        </div>
        <Button onClick={handleCalculate} className="flex items-center gap-2">
          <Calculator className="h-4 w-4" />
          Calculate Emissions
        </Button>
        
        <div className="space-y-2">
          <h4 className="font-semibold">Scope 1 Calculations</h4>
          {scope1Calculations.map((calc, index) => (
            <Card key={index} className="p-3">
              <div className="flex justify-between items-center">
                <div>
                  <p className="font-medium">{calc.activity}</p>
                  <p className="text-sm text-gray-600">
                    {calc.amount} × {calc.emissionFactor} = {calc.co2Equivalent.toFixed(2)} kg CO2e
                  </p>
                </div>
                <Badge variant="outline">{calc.source}</Badge>
              </div>
            </Card>
          ))}
          <div className="bg-green-50 p-3 rounded-lg">
            <p className="font-semibold">Total Scope 1 Emissions: {getTotalEmissions(scope1Calculations).toFixed(2)} kg CO2e</p>
          </div>
        </div>
      </div>
    );
  };

  const Scope2Calculator = () => {
    const [activity, setActivity] = useState('');
    const [amount, setAmount] = useState('');
    const [energyType, setEnergyType] = useState('electricity');

    const handleCalculate = () => {
      if (activity && amount) {
        addCalculation('scope2', activity, parseFloat(amount), 
          emissionFactors[energyType as keyof typeof emissionFactors].unit, energyType);
        setActivity('');
        setAmount('');
      }
    };

    return (
      <div className="space-y-4">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div>
            <Label>Activity Description</Label>
            <Input 
              value={activity}
              onChange={(e) => setActivity(e.target.value)}
              placeholder="e.g., Office electricity"
            />
          </div>
          <div>
            <Label>Energy Type</Label>
            <Select value={energyType} onValueChange={setEnergyType}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="electricity">Electricity</SelectItem>
                <SelectItem value="steam">Steam</SelectItem>
              </SelectContent>
            </Select>
          </div>
          <div>
            <Label>Amount (kWh)</Label>
            <Input 
              type="number"
              value={amount}
              onChange={(e) => setAmount(e.target.value)}
              placeholder="Enter kWh consumed"
            />
          </div>
        </div>
        <Button onClick={handleCalculate} className="flex items-center gap-2">
          <Zap className="h-4 w-4" />
          Calculate Emissions
        </Button>
        
        <div className="space-y-2">
          <h4 className="font-semibold">Scope 2 Calculations</h4>
          {scope2Calculations.map((calc, index) => (
            <Card key={index} className="p-3">
              <div className="flex justify-between items-center">
                <div>
                  <p className="font-medium">{calc.activity}</p>
                  <p className="text-sm text-gray-600">
                    {calc.amount} × {calc.emissionFactor} = {calc.co2Equivalent.toFixed(2)} kg CO2e
                  </p>
                </div>
                <Badge variant="outline">{calc.source}</Badge>
              </div>
            </Card>
          ))}
          <div className="bg-blue-50 p-3 rounded-lg">
            <p className="font-semibold">Total Scope 2 Emissions: {getTotalEmissions(scope2Calculations).toFixed(2)} kg CO2e</p>
          </div>
        </div>
      </div>
    );
  };

  const Scope3Calculator = () => {
    const [activity, setActivity] = useState('');
    const [amount, setAmount] = useState('');
    const [activityType, setActivityType] = useState('businessTravel');

    const handleCalculate = () => {
      if (activity && amount) {
        addCalculation('scope3', activity, parseFloat(amount), 
          emissionFactors[activityType as keyof typeof emissionFactors].unit, activityType);
        setActivity('');
        setAmount('');
      }
    };

    return (
      <div className="space-y-4">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div>
            <Label>Activity Description</Label>
            <Input 
              value={activity}
              onChange={(e) => setActivity(e.target.value)}
              placeholder="e.g., Business flights"
            />
          </div>
          <div>
            <Label>Activity Type</Label>
            <Select value={activityType} onValueChange={setActivityType}>
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="businessTravel">Business Travel</SelectItem>
                <SelectItem value="employeeCommuting">Employee Commuting</SelectItem>
                <SelectItem value="wasteDisposal">Waste Disposal</SelectItem>
                <SelectItem value="waterConsumption">Water Consumption</SelectItem>
              </SelectContent>
            </Select>
          </div>
          <div>
            <Label>Amount</Label>
            <Input 
              type="number"
              value={amount}
              onChange={(e) => setAmount(e.target.value)}
              placeholder="Enter quantity"
            />
          </div>
        </div>
        <Button onClick={handleCalculate} className="flex items-center gap-2">
          <Car className="h-4 w-4" />
          Calculate Emissions
        </Button>
        
        <div className="space-y-2">
          <h4 className="font-semibold">Scope 3 Calculations</h4>
          {scope3Calculations.map((calc, index) => (
            <Card key={index} className="p-3">
              <div className="flex justify-between items-center">
                <div>
                  <p className="font-medium">{calc.activity}</p>
                  <p className="text-sm text-gray-600">
                    {calc.amount} × {calc.emissionFactor} = {calc.co2Equivalent.toFixed(2)} kg CO2e
                  </p>
                </div>
                <Badge variant="outline">{calc.source}</Badge>
              </div>
            </Card>
          ))}
          <div className="bg-purple-50 p-3 rounded-lg">
            <p className="font-semibold">Total Scope 3 Emissions: {getTotalEmissions(scope3Calculations).toFixed(2)} kg CO2e</p>
          </div>
        </div>
      </div>
    );
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calculator className="h-5 w-5" />
            Carbon Accounting & Emission Calculators
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
            <Card className="p-4 bg-green-50">
              <div className="flex items-center gap-2">
                <Building className="h-5 w-5 text-green-600" />
                <div>
                  <p className="text-sm text-gray-600">Scope 1</p>
                  <p className="text-xl font-bold">{getTotalEmissions(scope1Calculations).toFixed(2)}</p>
                  <p className="text-sm">kg CO2e</p>
                </div>
              </div>
            </Card>
            <Card className="p-4 bg-blue-50">
              <div className="flex items-center gap-2">
                <Zap className="h-5 w-5 text-blue-600" />
                <div>
                  <p className="text-sm text-gray-600">Scope 2</p>
                  <p className="text-xl font-bold">{getTotalEmissions(scope2Calculations).toFixed(2)}</p>
                  <p className="text-sm">kg CO2e</p>
                </div>
              </div>
            </Card>
            <Card className="p-4 bg-purple-50">
              <div className="flex items-center gap-2">
                <Car className="h-5 w-5 text-purple-600" />
                <div>
                  <p className="text-sm text-gray-600">Scope 3</p>
                  <p className="text-xl font-bold">{getTotalEmissions(scope3Calculations).toFixed(2)}</p>
                  <p className="text-sm">kg CO2e</p>
                </div>
              </div>
            </Card>
            <Card className="p-4 bg-gray-50">
              <div className="flex items-center gap-2">
                <Target className="h-5 w-5 text-gray-600" />
                <div>
                  <p className="text-sm text-gray-600">Total</p>
                  <p className="text-xl font-bold">{getAllEmissions().toFixed(2)}</p>
                  <p className="text-sm">kg CO2e</p>
                </div>
              </div>
            </Card>
          </div>

          <Tabs defaultValue="scope1">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="scope1">Scope 1 - Direct</TabsTrigger>
              <TabsTrigger value="scope2">Scope 2 - Indirect Energy</TabsTrigger>
              <TabsTrigger value="scope3">Scope 3 - Other Indirect</TabsTrigger>
            </TabsList>
            
            <TabsContent value="scope1">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Scope 1: Direct GHG Emissions</CardTitle>
                  <p className="text-sm text-gray-600">
                    Direct emissions from sources owned or controlled by your organization
                  </p>
                </CardHeader>
                <CardContent>
                  <Scope1Calculator />
                </CardContent>
              </Card>
            </TabsContent>
            
            <TabsContent value="scope2">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Scope 2: Indirect GHG Emissions from Energy</CardTitle>
                  <p className="text-sm text-gray-600">
                    Indirect emissions from purchased electricity, steam, heating, and cooling
                  </p>
                </CardHeader>
                <CardContent>
                  <Scope2Calculator />
                </CardContent>
              </Card>
            </TabsContent>
            
            <TabsContent value="scope3">
              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Scope 3: Other Indirect GHG Emissions</CardTitle>
                  <p className="text-sm text-gray-600">
                    All other indirect emissions in your value chain
                  </p>
                </CardHeader>
                <CardContent>
                  <Scope3Calculator />
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>
    </div>
  );
};
