import React, { useState, useMemo } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Separator } from '@/components/ui/separator';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Checkbox } from '@/components/ui/checkbox';
import { FileText, Download, Eye, Filter, Calendar, Globe, Users, Shield, Target } from 'lucide-react';
import { sustainabilityMetrics } from '@/data/sustainabilityMetrics';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

export const ReportBuilder = () => {
  const [selectedStandards, setSelectedStandards] = useState<string[]>(['GRI']);
  const [selectedCategories, setSelectedCategories] = useState<string[]>(['Environmental']);
  const [selectedScopes, setSelectedScopes] = useState<string[]>(['Scope 1']);
  const [reportName, setReportName] = useState('');
  const [startDate, setStartDate] = useState('2024-01-01');
  const [endDate, setEndDate] = useState('2024-12-31');
  const [showPreview, setShowPreview] = useState(false);
  const [userDataPoints, setUserDataPoints] = useState<any[]>([]);
  const { toast } = useToast();

  const standards = ['GRI', 'SASB', 'TCFD', 'IFRS S2', 'CDP', 'SDG'];
  const categories = [...new Set(sustainabilityMetrics.map(m => m.category))];
  const scopes = ['Scope 1', 'Scope 2', 'Scope 3'];

  // Filter metrics based on selections
  const filteredMetrics = useMemo(() => {
    return sustainabilityMetrics.filter(metric => {
      const matchesStandards = selectedStandards.some(standard => 
        metric.standardCodes[standard] && metric.standardCodes[standard].length > 0
      );
      const matchesCategories = selectedCategories.length === 0 || selectedCategories.includes(metric.category);
      const matchesScopes = selectedScopes.length === 0 || !metric.scope || selectedScopes.some(scope => 
        metric.scope === scope.toLowerCase().replace(' ', '-')
      );
      
      return matchesStandards && matchesCategories && matchesScopes;
    });
  }, [selectedStandards, selectedCategories, selectedScopes]);

  const handleStandardChange = (standard: string, checked: boolean) => {
    if (checked) {
      setSelectedStandards([...selectedStandards, standard]);
    } else {
      setSelectedStandards(selectedStandards.filter(s => s !== standard));
    }
  };

  const handleCategoryChange = (category: string, checked: boolean) => {
    if (checked) {
      setSelectedCategories([...selectedCategories, category]);
    } else {
      setSelectedCategories(selectedCategories.filter(c => c !== category));
    }
  };

  const handleScopeChange = (scope: string, checked: boolean) => {
    if (checked) {
      setSelectedScopes([...selectedScopes, scope]);
    } else {
      setSelectedScopes(selectedScopes.filter(s => s !== scope));
    }
  };

  const fetchUserData = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from('esg_data_points')
        .select('*')
        .eq('user_id', user.id)
        .gte('reporting_period', startDate)
        .lte('reporting_period', endDate);

      if (error) throw error;
      setUserDataPoints(data || []);
    } catch (error) {
      console.error('Error fetching user data:', error);
    }
  };

  const handleGenerateReport = async () => {
    if (!reportName.trim()) {
      toast({
        title: "Report Name Required",
        description: "Please enter a name for your report",
        variant: "destructive"
      });
      return;
    }

    await fetchUserData();
    setShowPreview(true);
  };

  const getCategoryIcon = (category: string) => {
    switch (category) {
      case 'Environmental': return <Globe className="h-4 w-4" />;
      case 'Social': return <Users className="h-4 w-4" />;
      case 'Governance': return <Shield className="h-4 w-4" />;
      case 'Economic': return <Target className="h-4 w-4" />;
      default: return <Globe className="h-4 w-4" />;
    }
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <FileText className="h-5 w-5" />
            ESG Report Builder
          </CardTitle>
          <p className="text-gray-600">
            Create comprehensive sustainability reports with multi-standard compliance and live data preview
          </p>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Report Configuration */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-4">
              <div>
                <Label htmlFor="reportName">Report Name</Label>
                <Input
                  id="reportName"
                  value={reportName}
                  onChange={(e) => setReportName(e.target.value)}
                  placeholder="e.g., 2024 Annual Sustainability Report"
                />
              </div>
              
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="startDate">Start Date</Label>
                  <Input
                    id="startDate"
                    type="date"
                    value={startDate}
                    onChange={(e) => setStartDate(e.target.value)}
                  />
                </div>
                <div>
                  <Label htmlFor="endDate">End Date</Label>
                  <Input
                    id="endDate"
                    type="date"
                    value={endDate}
                    onChange={(e) => setEndDate(e.target.value)}
                  />
                </div>
              </div>
            </div>

            {/* Standards Selection */}
            <div>
              <Label>Reporting Standards</Label>
              <div className="grid grid-cols-2 gap-2 mt-2">
                {standards.map(standard => (
                  <div key={standard} className="flex items-center space-x-2">
                    <Checkbox 
                      id={standard}
                      checked={selectedStandards.includes(standard)}
                      onCheckedChange={(checked) => handleStandardChange(standard, !!checked)}
                    />
                    <Label htmlFor={standard} className="flex items-center gap-2">
                      <Badge variant="outline" className="text-xs">{standard}</Badge>
                    </Label>
                  </div>
                ))}
              </div>
            </div>
          </div>

          <Separator />

          {/* Filters */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <Label>ESG Categories</Label>
              <div className="space-y-2 mt-2">
                {categories.map(category => (
                  <div key={category} className="flex items-center space-x-2">
                    <Checkbox 
                      id={category}
                      checked={selectedCategories.includes(category)}
                      onCheckedChange={(checked) => handleCategoryChange(category, !!checked)}
                    />
                    <Label htmlFor={category} className="flex items-center gap-2">
                      {getCategoryIcon(category)}
                      {category}
                    </Label>
                  </div>
                ))}
              </div>
            </div>

            <div>
              <Label>Carbon Scopes</Label>
              <div className="space-y-2 mt-2">
                {scopes.map(scope => (
                  <div key={scope} className="flex items-center space-x-2">
                    <Checkbox 
                      id={scope}
                      checked={selectedScopes.includes(scope)}
                      onCheckedChange={(checked) => handleScopeChange(scope, !!checked)}
                    />
                    <Label htmlFor={scope}>{scope}</Label>
                  </div>
                ))}
              </div>
            </div>
          </div>

          <Separator />

          {/* Actions */}
          <div className="flex gap-4">
            <Button onClick={handleGenerateReport} className="flex items-center gap-2">
              <Eye className="h-4 w-4" />
              Generate Report Preview
            </Button>
            <Button variant="outline" className="flex items-center gap-2">
              <Download className="h-4 w-4" />
              Export Report
            </Button>
          </div>

          {/* Metrics Preview */}
          <div className="mt-6">
            <h3 className="text-lg font-semibold mb-4">
              Report will include {filteredMetrics.length} metrics
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {filteredMetrics.slice(0, 6).map(metric => (
                <Card key={metric.id} className="p-4">
                  <div className="flex items-center gap-2 mb-2">
                    {getCategoryIcon(metric.category)}
                    <span className="font-medium text-sm">{metric.name}</span>
                  </div>
                  <div className="text-xs text-gray-600 mb-2">{metric.subcategory}</div>
                  <div className="flex flex-wrap gap-1">
                    {metric.standards.slice(0, 2).map((standard, index) => (
                      <Badge key={`${standard.name}-${index}`} variant="outline" className="text-xs">
                        {standard.name} {standard.code}
                      </Badge>
                    ))}
                  </div>
                </Card>
              ))}
            </div>
            {filteredMetrics.length > 6 && (
              <p className="text-sm text-gray-600 mt-2">
                And {filteredMetrics.length - 6} more metrics...
              </p>
            )}
          </div>
        </CardContent>
      </Card>

      {/* Live Preview */}
      {showPreview && (
        <Card>
          <CardHeader>
            <CardTitle>Report Preview</CardTitle>
            <p className="text-gray-600">
              Live preview based on your organization's data for {startDate} to {endDate}
            </p>
          </CardHeader>
          <CardContent>
            <ScrollArea className="h-96">
              <div className="space-y-4">
                {filteredMetrics.map(metric => {
                  const userValue = userDataPoints.find(dp => dp.metric_name === metric.name);
                  return (
                    <div key={metric.id} className="border rounded-lg p-4">
                      <div className="flex justify-between items-start">
                        <div className="flex-1">
                          <h4 className="font-medium">{metric.name}</h4>
                          <p className="text-sm text-gray-600 mt-1">{metric.description}</p>
                          <div className="flex flex-wrap gap-2 mt-2">
                            {metric.standards.map((standard, index) => (
                              <Badge key={`${standard.name}-${index}`} variant="outline" className="text-xs">
                                {standard.name} {standard.code}
                              </Badge>
                            ))}
                          </div>
                        </div>
                        <div className="ml-4 text-right">
                          <div className="text-lg font-semibold">
                            {userValue?.metric_value || 'No data'}
                          </div>
                          <div className="text-sm text-gray-600">{metric.unit}</div>
                        </div>
                      </div>
                    </div>
                  );
                })}
              </div>
            </ScrollArea>
          </CardContent>
        </Card>
      )}
    </div>
  );
};
