
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { AlertTriangle, Plus, Edit, Trash2, Calendar, User } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface RiskAssessment {
  id: string;
  risk_category: string;
  risk_name: string;
  risk_description: string;
  likelihood: number;
  impact: number;
  risk_score: number;
  mitigation_strategy: string;
  responsible_person: string;
  review_date: string;
  status: string;
  created_at: string;
}

export const RiskManagementMain = () => {
  const [risks, setRisks] = useState<RiskAssessment[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [showAddForm, setShowAddForm] = useState(false);
  const [editingRisk, setEditingRisk] = useState<RiskAssessment | null>(null);
  const [filterCategory, setFilterCategory] = useState('all');
  const [filterStatus, setFilterStatus] = useState('all');
  const { toast } = useToast();

  const riskCategories = [
    { value: 'climate', label: 'Climate Risk', color: 'bg-red-100 text-red-800' },
    { value: 'regulatory', label: 'Regulatory Risk', color: 'bg-blue-100 text-blue-800' },
    { value: 'operational', label: 'Operational Risk', color: 'bg-yellow-100 text-yellow-800' },
    { value: 'financial', label: 'Financial Risk', color: 'bg-green-100 text-green-800' },
    { value: 'reputational', label: 'Reputational Risk', color: 'bg-purple-100 text-purple-800' }
  ];

  const statusOptions = [
    { value: 'active', label: 'Active', color: 'bg-red-100 text-red-800' },
    { value: 'mitigated', label: 'Mitigated', color: 'bg-yellow-100 text-yellow-800' },
    { value: 'closed', label: 'Closed', color: 'bg-green-100 text-green-800' }
  ];

  useEffect(() => {
    fetchRisks();
  }, []);

  const fetchRisks = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from('risk_assessments')
        .select('*')
        .eq('user_id', user.id)
        .order('risk_score', { ascending: false });

      if (error) throw error;
      setRisks(data || []);
    } catch (error) {
      console.error('Error fetching risks:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const getRiskLevel = (score: number) => {
    if (score >= 20) return { level: 'Critical', color: 'bg-red-500 text-white' };
    if (score >= 15) return { level: 'High', color: 'bg-orange-500 text-white' };
    if (score >= 10) return { level: 'Medium', color: 'bg-yellow-500 text-white' };
    if (score >= 5) return { level: 'Low', color: 'bg-blue-500 text-white' };
    return { level: 'Very Low', color: 'bg-green-500 text-white' };
  };

  const filteredRisks = risks.filter(risk => {
    const categoryMatch = filterCategory === 'all' || risk.risk_category === filterCategory;
    const statusMatch = filterStatus === 'all' || risk.status === filterStatus;
    return categoryMatch && statusMatch;
  });

  const RiskForm = ({ risk }: { risk?: RiskAssessment }) => {
    const [formData, setFormData] = useState({
      risk_category: risk?.risk_category || 'climate',
      risk_name: risk?.risk_name || '',
      risk_description: risk?.risk_description || '',
      likelihood: risk?.likelihood || 1,
      impact: risk?.impact || 1,
      mitigation_strategy: risk?.mitigation_strategy || '',
      responsible_person: risk?.responsible_person || '',
      review_date: risk?.review_date || '',
      status: risk?.status || 'active'
    });

    const handleSubmit = async (e: React.FormEvent) => {
      e.preventDefault();
      
      try {
        const { data: { user } } = await supabase.auth.getUser();
        if (!user) throw new Error('User not authenticated');

        if (risk) {
          // Update existing risk
          const { error } = await supabase
            .from('risk_assessments')
            .update(formData)
            .eq('id', risk.id);

          if (error) throw error;
          toast({ title: "Risk Updated", description: "Risk assessment has been updated successfully" });
        } else {
          // Create new risk
          const { error } = await supabase
            .from('risk_assessments')
            .insert({
              ...formData,
              user_id: user.id
            });

          if (error) throw error;
          toast({ title: "Risk Created", description: "New risk assessment has been created successfully" });
        }

        fetchRisks();
        setShowAddForm(false);
        setEditingRisk(null);
      } catch (error) {
        console.error('Error saving risk:', error);
        toast({
          title: "Error",
          description: "Failed to save risk assessment",
          variant: "destructive"
        });
      }
    };

    return (
      <Card className="mb-6">
        <CardHeader>
          <CardTitle>{risk ? 'Edit Risk Assessment' : 'Add New Risk Assessment'}</CardTitle>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label>Risk Category</Label>
                <Select value={formData.risk_category} onValueChange={(value) => setFormData({...formData, risk_category: value})}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {riskCategories.map(cat => (
                      <SelectItem key={cat.value} value={cat.value}>{cat.label}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label>Risk Name</Label>
                <Input
                  value={formData.risk_name}
                  onChange={(e) => setFormData({...formData, risk_name: e.target.value})}
                  placeholder="e.g., Extreme weather events"
                  required
                />
              </div>
            </div>

            <div>
              <Label>Risk Description</Label>
              <Textarea
                value={formData.risk_description}
                onChange={(e) => setFormData({...formData, risk_description: e.target.value})}
                placeholder="Detailed description of the risk..."
                rows={3}
              />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <Label>Likelihood (1-5 scale)</Label>
                <Select value={formData.likelihood.toString()} onValueChange={(value) => setFormData({...formData, likelihood: parseInt(value)})}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1 - Very Unlikely</SelectItem>
                    <SelectItem value="2">2 - Unlikely</SelectItem>
                    <SelectItem value="3">3 - Possible</SelectItem>
                    <SelectItem value="4">4 - Likely</SelectItem>
                    <SelectItem value="5">5 - Very Likely</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <Label>Impact (1-5 scale)</Label>
                <Select value={formData.impact.toString()} onValueChange={(value) => setFormData({...formData, impact: parseInt(value)})}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1 - Minimal</SelectItem>
                    <SelectItem value="2">2 - Minor</SelectItem>
                    <SelectItem value="3">3 - Moderate</SelectItem>
                    <SelectItem value="4">4 - Major</SelectItem>
                    <SelectItem value="5">5 - Severe</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div>
              <Label>Risk Score: {formData.likelihood * formData.impact}</Label>
              <div className="mt-2">
                <Badge className={getRiskLevel(formData.likelihood * formData.impact).color}>
                  {getRiskLevel(formData.likelihood * formData.impact).level}
                </Badge>
              </div>
            </div>

            <div>
              <Label>Mitigation Strategy</Label>
              <Textarea
                value={formData.mitigation_strategy}
                onChange={(e) => setFormData({...formData, mitigation_strategy: e.target.value})}
                placeholder="Describe the mitigation strategy..."
                rows={3}
              />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div>
                <Label>Responsible Person</Label>
                <Input
                  value={formData.responsible_person}
                  onChange={(e) => setFormData({...formData, responsible_person: e.target.value})}
                  placeholder="e.g., John Smith"
                />
              </div>
              <div>
                <Label>Review Date</Label>
                <Input
                  type="date"
                  value={formData.review_date}
                  onChange={(e) => setFormData({...formData, review_date: e.target.value})}
                />
              </div>
              <div>
                <Label>Status</Label>
                <Select value={formData.status} onValueChange={(value) => setFormData({...formData, status: value})}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {statusOptions.map(status => (
                      <SelectItem key={status.value} value={status.value}>{status.label}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="flex gap-2">
              <Button type="submit">
                {risk ? 'Update Risk' : 'Create Risk'}
              </Button>
              <Button 
                type="button" 
                variant="outline" 
                onClick={() => {
                  setShowAddForm(false);
                  setEditingRisk(null);
                }}
              >
                Cancel
              </Button>
            </div>
          </form>
        </CardContent>
      </Card>
    );
  };

  const deleteRisk = async (riskId: string) => {
    if (!confirm('Are you sure you want to delete this risk assessment?')) return;

    try {
      const { error } = await supabase
        .from('risk_assessments')
        .delete()
        .eq('id', riskId);

      if (error) throw error;

      toast({ title: "Risk Deleted", description: "Risk assessment has been deleted successfully" });
      fetchRisks();
    } catch (error) {
      console.error('Error deleting risk:', error);
      toast({
        title: "Error",
        description: "Failed to delete risk assessment",
        variant: "destructive"
      });
    }
  };

  if (isLoading) {
    return <div className="p-6">Loading risk assessments...</div>;
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Risk Management</h1>
          <p className="text-gray-600 mt-2">
            Comprehensive ESG risk assessment and management system
          </p>
        </div>
        <Button onClick={() => setShowAddForm(true)} className="flex items-center gap-2">
          <Plus className="h-4 w-4" />
          Add Risk Assessment
        </Button>
      </div>

      {/* Risk Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="p-4">
          <div className="flex items-center gap-2">
            <AlertTriangle className="h-5 w-5 text-red-500" />
            <div>
              <p className="text-sm text-gray-600">Critical Risks</p>
              <p className="text-xl font-bold">{risks.filter(r => r.risk_score >= 20).length}</p>
            </div>
          </div>
        </Card>
        <Card className="p-4">
          <div className="flex items-center gap-2">
            <AlertTriangle className="h-5 w-5 text-orange-500" />
            <div>
              <p className="text-sm text-gray-600">High Risks</p>
              <p className="text-xl font-bold">{risks.filter(r => r.risk_score >= 15 && r.risk_score < 20).length}</p>
            </div>
          </div>
        </Card>
        <Card className="p-4">
          <div className="flex items-center gap-2">
            <AlertTriangle className="h-5 w-5 text-yellow-500" />
            <div>
              <p className="text-sm text-gray-600">Medium Risks</p>
              <p className="text-xl font-bold">{risks.filter(r => r.risk_score >= 10 && r.risk_score < 15).length}</p>
            </div>
          </div>
        </Card>
        <Card className="p-4">
          <div className="flex items-center gap-2">
            <AlertTriangle className="h-5 w-5 text-green-500" />
            <div>
              <p className="text-sm text-gray-600">Total Risks</p>
              <p className="text-xl font-bold">{risks.length}</p>
            </div>
          </div>
        </Card>
      </div>

      {/* Add/Edit Form */}
      {(showAddForm || editingRisk) && <RiskForm risk={editingRisk || undefined} />}

      {/* Filters */}
      <Card>
        <CardHeader>
          <CardTitle>Risk Assessment Filter</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <Label>Filter by Category</Label>
              <Select value={filterCategory} onValueChange={setFilterCategory}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Categories</SelectItem>
                  {riskCategories.map(cat => (
                    <SelectItem key={cat.value} value={cat.value}>{cat.label}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label>Filter by Status</Label>
              <Select value={filterStatus} onValueChange={setFilterStatus}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Statuses</SelectItem>
                  {statusOptions.map(status => (
                    <SelectItem key={status.value} value={status.value}>{status.label}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Risk List */}
      <div className="space-y-4">
        {filteredRisks.map((risk) => {
          const riskLevel = getRiskLevel(risk.risk_score);
          const category = riskCategories.find(c => c.value === risk.risk_category);
          const status = statusOptions.find(s => s.value === risk.status);

          return (
            <Card key={risk.id}>
              <CardContent className="p-6">
                <div className="flex justify-between items-start mb-4">
                  <div>
                    <h3 className="text-lg font-semibold">{risk.risk_name}</h3>
                    <p className="text-gray-600 mt-1">{risk.risk_description}</p>
                  </div>
                  <div className="flex gap-2">
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => setEditingRisk(risk)}
                    >
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button
                      size="sm"
                      variant="outline"
                      onClick={() => deleteRisk(risk.id)}
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-4">
                  <div>
                    <p className="text-sm font-medium">Category</p>
                    <Badge className={category?.color}>{category?.label}</Badge>
                  </div>
                  <div>
                    <p className="text-sm font-medium">Risk Score</p>
                    <Badge className={riskLevel.color}>
                      {risk.risk_score} ({riskLevel.level})
                    </Badge>
                  </div>
                  <div>
                    <p className="text-sm font-medium">Status</p>
                    <Badge className={status?.color}>{status?.label}</Badge>
                  </div>
                  <div>
                    <p className="text-sm font-medium">Likelihood × Impact</p>
                    <p className="text-sm">{risk.likelihood} × {risk.impact}</p>
                  </div>
                </div>

                {risk.mitigation_strategy && (
                  <div className="mb-4">
                    <p className="text-sm font-medium mb-1">Mitigation Strategy</p>
                    <p className="text-sm text-gray-600">{risk.mitigation_strategy}</p>
                  </div>
                )}

                <div className="flex justify-between text-sm text-gray-500">
                  <div className="flex items-center gap-4">
                    {risk.responsible_person && (
                      <div className="flex items-center gap-1">
                        <User className="h-4 w-4" />
                        <span>{risk.responsible_person}</span>
                      </div>
                    )}
                    {risk.review_date && (
                      <div className="flex items-center gap-1">
                        <Calendar className="h-4 w-4" />
                        <span>Review: {new Date(risk.review_date).toLocaleDateString()}</span>
                      </div>
                    )}
                  </div>
                  <span>Created: {new Date(risk.created_at).toLocaleDateString()}</span>
                </div>
              </CardContent>
            </Card>
          );
        })}
      </div>

      {filteredRisks.length === 0 && (
        <Card>
          <CardContent className="p-12 text-center">
            <AlertTriangle className="h-12 w-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No Risk Assessments Found</h3>
            <p className="text-gray-600 mb-4">
              {risks.length === 0 
                ? "Start by adding your first risk assessment to track and manage ESG risks."
                : "No risks match the current filter criteria. Try adjusting your filters."
              }
            </p>
            {risks.length === 0 && (
              <Button onClick={() => setShowAddForm(true)}>
                <Plus className="h-4 w-4 mr-2" />
                Add First Risk Assessment
              </Button>
            )}
          </CardContent>
        </Card>
      )}
    </div>
  );
};
