
import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { CheckCircle, XCircle, Loader2, AlertTriangle } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';

interface StorageConfig {
  provider: string;
  region: string;
  bucketName: string;
  accessKey: string;
  secretKey: string;
  endpoint: string;
}

interface ProfessionalStorageConnectionProps {
  onConfigSaved: (isConfigured: boolean) => void;
}

export const ProfessionalStorageConnection = ({ onConfigSaved }: ProfessionalStorageConnectionProps) => {
  const { toast } = useToast();
  const [storageConfig, setStorageConfig] = useState<StorageConfig>({
    provider: '',
    region: '',
    bucketName: '',
    accessKey: '',
    secretKey: '',
    endpoint: '',
  });
  const [isTesting, setIsTesting] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [testResult, setTestResult] = useState<'none' | 'success' | 'failed'>('none');
  const [testError, setTestError] = useState<string>('');

  useEffect(() => {
    // Load existing configuration if available
    const existingConfig = localStorage.getItem('cloudStorageConfig');
    if (existingConfig) {
      try {
        const config = JSON.parse(existingConfig);
        if (config.type === 'professional' && config.config) {
          setStorageConfig({
            ...config.config,
            // Don't load the actual secret key for security
            secretKey: config.config.secretKey ? '••••••••••••••••' : ''
          });
          setTestResult('success');
        }
      } catch (error) {
        console.error('Error loading existing configuration:', error);
      }
    }
  }, []);

  const validateConfiguration = (): { isValid: boolean; errors: string[] } => {
    const errors: string[] = [];

    if (!storageConfig.provider) errors.push('Storage provider is required');
    if (!storageConfig.bucketName) errors.push('Bucket/Container name is required');
    if (!storageConfig.accessKey) errors.push('Access Key ID is required');
    if (!storageConfig.secretKey || storageConfig.secretKey === '••••••••••••••••') {
      errors.push('Secret Access Key is required');
    }

    // Validate bucket name format (common rules across providers)
    if (storageConfig.bucketName && !storageConfig.bucketName.match(/^[a-z0-9.-]{3,63}$/)) {
      errors.push('Bucket name must be 3-63 characters, lowercase letters, numbers, dots, and hyphens only');
    }

    // Validate access key format (basic length check)
    if (storageConfig.accessKey && storageConfig.accessKey.length < 16) {
      errors.push('Access Key ID appears to be too short');
    }

    // Validate secret key format (basic length check)
    if (storageConfig.secretKey && storageConfig.secretKey !== '••••••••••••••••' && storageConfig.secretKey.length < 32) {
      errors.push('Secret Access Key appears to be too short');
    }

    // Provider-specific validations
    switch (storageConfig.provider) {
      case 'aws':
        if (!storageConfig.region) errors.push('AWS region is required');
        if (storageConfig.region && !storageConfig.region.match(/^[a-z]{2}-[a-z]+-\d{1}$/)) {
          errors.push('AWS region format appears invalid (e.g., us-east-1)');
        }
        break;
      case 'gcp':
        if (storageConfig.accessKey && !storageConfig.accessKey.startsWith('GOOG')) {
          errors.push('Google Cloud access keys typically start with "GOOG"');
        }
        break;
      case 'azure':
        if (!storageConfig.endpoint && !storageConfig.region) {
          errors.push('Azure requires either a custom endpoint or region');
        }
        break;
    }

    return {
      isValid: errors.length === 0,
      errors
    };
  };

  const testConnection = async () => {
    setIsTesting(true);
    setTestResult('none');
    setTestError('');
    
    try {
      const validation = validateConfiguration();
      if (!validation.isValid) {
        setTestError(validation.errors.join(', '));
        setTestResult('failed');
        toast({
          title: "Validation Error",
          description: validation.errors[0],
          variant: "destructive",
        });
        return;
      }

      // Simulate API call to test connection
      await new Promise(resolve => setTimeout(resolve, 3000));
      
      // For now, we'll simulate the test since real cloud API calls require backend
      // In a real implementation, this would make actual API calls to test credentials
      
      toast({
        title: "Test Connection Note",
        description: "Connection testing requires backend API integration. Configuration has been validated for format correctness.",
        variant: "default",
      });

      setTestResult('success');
      setTestError('');
      
    } catch (error) {
      const errorMessage = error instanceof Error ? error.message : 'Unknown error occurred';
      setTestResult('failed');
      setTestError(errorMessage);
      toast({
        title: "Connection Test Failed",
        description: errorMessage,
        variant: "destructive",
      });
    } finally {
      setIsTesting(false);
    }
  };

  const saveConfiguration = async () => {
    setIsSaving(true);
    
    try {
      const validation = validateConfiguration();
      if (!validation.isValid) {
        toast({
          title: "Validation Error",
          description: validation.errors[0],
          variant: "destructive",
        });
        return;
      }

      // Create configuration object (in production, this would be encrypted)
      const configToSave = {
        type: 'professional',
        config: {
          ...storageConfig,
          // In production, encrypt the secret key before storing
          secretKey: storageConfig.secretKey === '••••••••••••••••' ? 'EXISTING_KEY_PRESERVED' : 'KEY_WOULD_BE_ENCRYPTED',
          setupDate: new Date().toISOString()
        },
        timestamp: new Date().toISOString(),
      };

      // Save configuration
      localStorage.setItem('cloudStorageConfig', JSON.stringify(configToSave));

      // Log configuration details (excluding sensitive data)
      console.log('Professional storage configuration saved:', {
        provider: storageConfig.provider,
        region: storageConfig.region,
        bucketName: storageConfig.bucketName,
        endpoint: storageConfig.endpoint,
        hasAccessKey: !!storageConfig.accessKey,
        hasSecretKey: !!storageConfig.secretKey && storageConfig.secretKey !== '••••••••••••••••'
      });

      onConfigSaved(true);
      toast({
        title: "Configuration Saved Successfully",
        description: "Your professional cloud storage settings have been configured and validated.",
      });

    } catch (error) {
      toast({
        title: "Save Failed",
        description: "Failed to save configuration. Please try again.",
        variant: "destructive",
      });
    } finally {
      setIsSaving(false);
    }
  };

  const getProviderName = (provider: string) => {
    const names: { [key: string]: string } = {
      'aws': 'Amazon S3',
      'gcp': 'Google Cloud Storage',
      'azure': 'Azure Blob Storage',
      'digitalocean': 'DigitalOcean Spaces',
      'wasabi': 'Wasabi Cloud Storage',
    };
    return names[provider] || provider;
  };

  const getTestIcon = () => {
    if (isTesting) return <Loader2 className="h-4 w-4 animate-spin" />;
    if (testResult === 'success') return <CheckCircle className="h-4 w-4 text-green-600" />;
    if (testResult === 'failed') return <XCircle className="h-4 w-4 text-red-600" />;
    return null;
  };

  const validation = validateConfiguration();

  return (
    <div className="space-y-6">
      <Alert>
        <AlertTriangle className="h-4 w-4" />
        <AlertDescription>
          <strong>Professional Setup:</strong> This configuration validates your settings and prepares them for backend integration. 
          Real cloud API connections require server-side implementation for security.
        </AlertDescription>
      </Alert>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        <div>
          <Label htmlFor="provider">Storage Provider *</Label>
          <Select
            value={storageConfig.provider}
            onValueChange={(value) => {
              setStorageConfig(prev => ({ ...prev, provider: value }));
              setTestResult('none');
              setTestError('');
            }}
          >
            <SelectTrigger>
              <SelectValue placeholder="Select your cloud provider" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="aws">Amazon S3</SelectItem>
              <SelectItem value="gcp">Google Cloud Storage</SelectItem>
              <SelectItem value="azure">Azure Blob Storage</SelectItem>
              <SelectItem value="digitalocean">DigitalOcean Spaces</SelectItem>
              <SelectItem value="wasabi">Wasabi Cloud Storage</SelectItem>
            </SelectContent>
          </Select>
        </div>

        <div>
          <Label htmlFor="region">Region {storageConfig.provider === 'aws' && '*'}</Label>
          <Input
            id="region"
            value={storageConfig.region}
            onChange={(e) => {
              setStorageConfig(prev => ({ ...prev, region: e.target.value }));
              setTestResult('none');
            }}
            placeholder="e.g., us-east-1"
          />
        </div>

        <div>
          <Label htmlFor="bucketName">Bucket/Container Name *</Label>
          <Input
            id="bucketName"
            value={storageConfig.bucketName}
            onChange={(e) => {
              setStorageConfig(prev => ({ ...prev, bucketName: e.target.value }));
              setTestResult('none');
            }}
            placeholder="your-greencard-data-bucket"
          />
        </div>

        <div>
          <Label htmlFor="endpoint">Custom Endpoint (Optional)</Label>
          <Input
            id="endpoint"
            value={storageConfig.endpoint}
            onChange={(e) => {
              setStorageConfig(prev => ({ ...prev, endpoint: e.target.value }));
              setTestResult('none');
            }}
            placeholder="https://your-custom-endpoint.com"
          />
        </div>

        <div>
          <Label htmlFor="accessKey">Access Key ID *</Label>
          <Input
            id="accessKey"
            type="password"
            value={storageConfig.accessKey}
            onChange={(e) => {
              setStorageConfig(prev => ({ ...prev, accessKey: e.target.value }));
              setTestResult('none');
            }}
            placeholder="Your access key"
          />
        </div>

        <div>
          <Label htmlFor="secretKey">Secret Access Key *</Label>
          <Input
            id="secretKey"
            type="password"
            value={storageConfig.secretKey}
            onChange={(e) => {
              setStorageConfig(prev => ({ ...prev, secretKey: e.target.value }));
              setTestResult('none');
            }}
            placeholder="Your secret key"
          />
        </div>
      </div>

      {!validation.isValid && (
        <Alert>
          <AlertTriangle className="h-4 w-4" />
          <AlertDescription>
            <strong>Validation Issues:</strong>
            <ul className="list-disc list-inside mt-2">
              {validation.errors.map((error, index) => (
                <li key={index} className="text-sm">{error}</li>
              ))}
            </ul>
          </AlertDescription>
        </Alert>
      )}

      {testResult !== 'none' && (
        <Alert className={testResult === 'success' ? 'border-green-200 bg-green-50' : 'border-red-200 bg-red-50'}>
          <AlertDescription className={testResult === 'success' ? 'text-green-800' : 'text-red-800'}>
            {testResult === 'success' 
              ? '✅ Configuration validated successfully! Settings are ready for backend integration.'
              : `❌ Configuration validation failed: ${testError}`
            }
          </AlertDescription>
        </Alert>
      )}

      <div className="flex gap-4">
        <Button 
          onClick={testConnection} 
          variant="outline"
          disabled={isTesting || !validation.isValid}
          className="flex items-center gap-2"
        >
          {getTestIcon()}
          {isTesting ? 'Validating...' : 'Test & Validate'}
        </Button>
        
        <Button 
          onClick={saveConfiguration}
          disabled={isSaving || !validation.isValid || testResult !== 'success'}
          className="flex items-center gap-2"
        >
          {isSaving && <Loader2 className="h-4 w-4 animate-spin" />}
          Save Configuration
        </Button>
      </div>

      <Alert>
        <AlertDescription>
          <strong>Security Note:</strong> In production, credentials are encrypted and stored securely. 
          Backend API integration is required for actual cloud storage operations.
        </AlertDescription>
      </Alert>
    </div>
  );
};
