
import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Card, CardContent } from '@/components/ui/card';
import { CheckCircle, XCircle, Loader2, HardDrive, AlertTriangle, Folder } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';
import { Alert, AlertDescription } from '@/components/ui/alert';

interface SimpleStorageConnectionProps {
  onConnectionStatusChange: (status: 'none' | 'connected' | 'failed') => void;
}

export const SimpleStorageConnection = ({ onConnectionStatusChange }: SimpleStorageConnectionProps) => {
  const { toast } = useToast();
  const [provider, setProvider] = useState('');
  const [folderName, setFolderName] = useState('GreenCard_Data');
  const [localStoragePath, setLocalStoragePath] = useState('');
  const [storageType, setStorageType] = useState<'browser' | 'local-server'>('browser');
  const [isConnecting, setIsConnecting] = useState(false);
  const [connectionStatus, setConnectionStatus] = useState<'none' | 'connected' | 'failed'>('none');
  const [storageInfo, setStorageInfo] = useState<{
    browserQuota: number;
    browserUsed: number;
    isQuotaReliable: boolean;
    estimatedCapacity: string;
  } | null>(null);

  useEffect(() => {
    // Check if there's an existing storage configuration
    const existingConfig = localStorage.getItem('cloudStorageConfig');
    if (existingConfig) {
      try {
        const config = JSON.parse(existingConfig);
        if (config.type === 'simple' && config.config) {
          setProvider(config.config.provider);
          setFolderName(config.config.folderName || 'GreenCard_Data');
          setLocalStoragePath(config.config.storagePath || '');
          setStorageType(config.config.storageType || 'browser');
          setConnectionStatus('connected');
          onConnectionStatusChange('connected');
        }
      } catch (error) {
        console.error('Error loading existing configuration:', error);
      }
    }

    // Get more accurate browser storage information
    checkBrowserStorageCapacity();
  }, [onConnectionStatusChange]);

  const checkBrowserStorageCapacity = async () => {
    try {
      if ('storage' in navigator && 'estimate' in navigator.storage) {
        const estimate = await navigator.storage.estimate();
        const quota = estimate.quota || 0;
        const usage = estimate.usage || 0;
        
        // Test actual localStorage capacity with a small test
        const testResult = await testActualLocalStorageCapacity();
        
        setStorageInfo({
          browserQuota: quota,
          browserUsed: usage,
          isQuotaReliable: quota > 0 && quota < 1000000000000, // Less than 1TB indicates more reliable estimate
          estimatedCapacity: testResult.estimatedCapacity
        });
      } else {
        // Fallback for browsers without storage estimation
        const testResult = await testActualLocalStorageCapacity();
        setStorageInfo({
          browserQuota: 0,
          browserUsed: 0,
          isQuotaReliable: false,
          estimatedCapacity: testResult.estimatedCapacity
        });
      }
    } catch (error) {
      console.error('Error checking storage capacity:', error);
      setStorageInfo({
        browserQuota: 0,
        browserUsed: 0,
        isQuotaReliable: false,
        estimatedCapacity: 'Unknown - Browser storage estimation failed'
      });
    }
  };

  const testActualLocalStorageCapacity = async (): Promise<{
    estimatedCapacity: string;
    canStoreData: boolean;
  }> => {
    try {
      // Test with progressively larger chunks to estimate capacity
      const testSizes = [1024, 10240, 102400, 1048576]; // 1KB, 10KB, 100KB, 1MB
      let maxSuccessfulSize = 0;
      
      for (const size of testSizes) {
        try {
          const testData = 'x'.repeat(size);
          const testKey = `capacity_test_${Date.now()}_${size}`;
          
          localStorage.setItem(testKey, testData);
          localStorage.removeItem(testKey);
          maxSuccessfulSize = size;
        } catch (error) {
          break;
        }
      }
      
      if (maxSuccessfulSize >= 1048576) {
        return {
          estimatedCapacity: 'Good (Can handle large ESG datasets)',
          canStoreData: true
        };
      } else if (maxSuccessfulSize >= 102400) {
        return {
          estimatedCapacity: 'Moderate (Suitable for basic ESG data)',
          canStoreData: true
        };
      } else if (maxSuccessfulSize >= 10240) {
        return {
          estimatedCapacity: 'Limited (May need frequent cleanup)',
          canStoreData: true
        };
      } else {
        return {
          estimatedCapacity: 'Very Limited (Not recommended for ESG data)',
          canStoreData: false
        };
      }
    } catch (error) {
      return {
        estimatedCapacity: 'Unable to test storage capacity',
        canStoreData: false
      };
    }
  };

  const validateLocalServerPath = (path: string): { isValid: boolean; message: string } => {
    if (!path || path.trim() === '') {
      return { isValid: false, message: 'Path cannot be empty' };
    }
    
    const trimmedPath = path.trim();
    
    // Basic path validation patterns
    const windowsPathRegex = /^[A-Za-z]:\\(?:[^<>:"|?*\r\n\/]+\\)*[^<>:"|?*\r\n\/]*$/;
    const unixPathRegex = /^\/(?:[^<>:"|?*\r\n\\]+\/)*[^<>:"|?*\r\n\\]*$/;
    const networkPathRegex = /^\\\\[^<>:"|?*\r\n\/]+\\(?:[^<>:"|?*\r\n\/]+\\)*[^<>:"|?*\r\n\/]*$/;
    
    if (windowsPathRegex.test(trimmedPath)) {
      return { isValid: true, message: 'Valid Windows path format' };
    } else if (unixPathRegex.test(trimmedPath)) {
      return { isValid: true, message: 'Valid Unix/Linux path format' };
    } else if (networkPathRegex.test(trimmedPath)) {
      return { isValid: true, message: 'Valid network path format' };
    } else {
      return { 
        isValid: false, 
        message: 'Invalid path format. Use: C:\\folder\\path (Windows), /home/user/folder (Unix), or \\\\server\\share (Network)' 
      };
    }
  };

  const handleConnection = async (selectedProvider: string) => {
    setIsConnecting(true);
    setConnectionStatus('none');
    onConnectionStatusChange('none');
    
    try {
      if (selectedProvider === 'localstorage') {
        await handleLocalStorageSetup();
      } else {
        await handleCloudProviderAuth(selectedProvider);
      }
    } catch (error) {
      setConnectionStatus('failed');
      onConnectionStatusChange('failed');
      toast({
        title: "Connection Error",
        description: error instanceof Error ? error.message : "Failed to setup storage. Please try again.",
        variant: "destructive",
      });
    } finally {
      setIsConnecting(false);
    }
  };

  const handleLocalStorageSetup = async () => {
    try {
      // Validate storage type and path
      if (storageType === 'local-server') {
        if (!localStoragePath) {
          throw new Error('Local server path is required when using local server storage');
        }
        
        const pathValidation = validateLocalServerPath(localStoragePath);
        if (!pathValidation.isValid) {
          throw new Error(pathValidation.message);
        }
      }

      // Test browser storage if using browser storage
      if (storageType === 'browser') {
        if (!window.localStorage) {
          throw new Error('Local storage is not available in this browser');
        }

        // Test actual storage capacity
        const capacityTest = await testActualLocalStorageCapacity();
        if (!capacityTest.canStoreData) {
          throw new Error('Insufficient browser storage capacity for ESG data. Consider using local server storage instead.');
        }
      }

      // Create storage structure
      const storageStructure = {
        folders: {
          [folderName]: {
            'carbon_data': {},
            'reports': {},
            'uploads': {},
            'exports': {},
            'backups': {}
          }
        },
        metadata: {
          created: new Date().toISOString(),
          provider: 'localstorage',
          storageType: storageType,
          path: storageType === 'local-server' ? localStoragePath : 'browser-storage',
          version: '1.0'
        }
      };
      
      localStorage.setItem('greencard_storage_structure', JSON.stringify(storageStructure));

      // Save configuration
      const config = {
        type: 'simple',
        config: { 
          provider: 'localstorage', 
          folderName,
          storagePath: storageType === 'local-server' ? localStoragePath : 'browser-storage',
          storageType: storageType,
          setupDate: new Date().toISOString(),
          validated: true
        },
        timestamp: new Date().toISOString(),
      };

      localStorage.setItem('cloudStorageConfig', JSON.stringify(config));
      
      // Test write/read functionality
      const testData = { 
        test: 'ESG data sample', 
        timestamp: new Date().toISOString(),
        storageType: storageType,
        path: storageType === 'local-server' ? localStoragePath : 'browser-storage'
      };
      
      const testKey = `${folderName}_storage_test`;
      localStorage.setItem(testKey, JSON.stringify(testData));
      const retrieved = localStorage.getItem(testKey);
      
      if (!retrieved || JSON.parse(retrieved).test !== testData.test) {
        throw new Error('Storage read/write validation failed');
      }
      
      // Clean up test data
      localStorage.removeItem(testKey);

      setConnectionStatus('connected');
      onConnectionStatusChange('connected');
      
      const storageDescription = storageType === 'local-server' 
        ? `at local path: ${localStoragePath}` 
        : 'in browser local storage';
      
      toast({
        title: "Local Storage Configured Successfully",
        description: `Your ESG data will be stored ${storageDescription}. Storage capacity: ${storageInfo?.estimatedCapacity || 'Unknown'}`,
      });

    } catch (error) {
      throw error;
    }
  };

  const handleCloudProviderAuth = async (selectedProvider: string) => {
    setConnectionStatus('failed');
    onConnectionStatusChange('failed');
    
    toast({
      title: "Cloud Provider Setup Required",
      description: `${getProviderName(selectedProvider)} integration requires backend OAuth configuration. Please use Local Storage for now or contact support for cloud setup.`,
      variant: "destructive",
    });
    
    throw new Error(`${getProviderName(selectedProvider)} requires additional backend configuration`);
  };

  const getProviderName = (provider: string) => {
    const names: { [key: string]: string } = {
      'googledrive': 'Google Drive',
      'dropbox': 'Dropbox',
      'onedrive': 'Microsoft OneDrive',
      'localstorage': 'Local Storage',
    };
    return names[provider] || provider;
  };

  const getConnectionIcon = () => {
    if (isConnecting) return <Loader2 className="h-4 w-4 animate-spin" />;
    if (connectionStatus === 'connected') return <CheckCircle className="h-4 w-4 text-green-600" />;
    if (connectionStatus === 'failed') return <XCircle className="h-4 w-4 text-red-600" />;
    return null;
  };

  const formatBytes = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  const pathValidation = localStoragePath ? validateLocalServerPath(localStoragePath) : null;

  return (
    <div className="space-y-4">
      <div>
        <Label htmlFor="simple-provider">Choose Your Storage Provider</Label>
        <Select
          value={provider}
          onValueChange={(value) => {
            setProvider(value);
            setConnectionStatus('none');
            onConnectionStatusChange('none');
          }}
        >
          <SelectTrigger>
            <SelectValue placeholder="Select your preferred storage" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="localstorage">Local Storage (Recommended)</SelectItem>
            <SelectItem value="googledrive">Google Drive (Requires Setup)</SelectItem>
            <SelectItem value="dropbox">Dropbox (Requires Setup)</SelectItem>
            <SelectItem value="onedrive">Microsoft OneDrive (Requires Setup)</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <div>
        <Label htmlFor="folderName">Data Folder Name</Label>
        <Input
          id="folderName"
          value={folderName}
          onChange={(e) => setFolderName(e.target.value)}
          placeholder="GreenCard_Data"
        />
        <p className="text-xs text-gray-500 mt-1">
          A folder structure with this name will organize your ESG data.
        </p>
      </div>

      {provider === 'localstorage' && (
        <>
          <div>
            <Label>Storage Location</Label>
            <Select
              value={storageType}
              onValueChange={(value: 'browser' | 'local-server') => {
                setStorageType(value);
                setConnectionStatus('none');
                onConnectionStatusChange('none');
              }}
            >
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="browser">Browser Storage</SelectItem>
                <SelectItem value="local-server">Local Server/Network Path</SelectItem>
              </SelectContent>
            </Select>
          </div>

          {storageType === 'local-server' && (
            <div>
              <Label htmlFor="localPath" className="flex items-center gap-2">
                <Folder className="h-4 w-4" />
                Local Server/Network Path
              </Label>
              <Input
                id="localPath"
                value={localStoragePath}
                onChange={(e) => {
                  setLocalStoragePath(e.target.value);
                  setConnectionStatus('none');
                  onConnectionStatusChange('none');
                }}
                placeholder="C:\ESG_Data or /home/user/esg_data or \\server\share\esg"
              />
              {pathValidation && (
                <p className={`text-xs mt-1 ${pathValidation.isValid ? 'text-green-600' : 'text-red-600'}`}>
                  {pathValidation.message}
                </p>
              )}
              <p className="text-xs text-gray-500 mt-1">
                Specify the exact path where your ESG data should be stored on your local system or network.
              </p>
            </div>
          )}

          {storageType === 'browser' && storageInfo && (
            <Alert>
              <AlertTriangle className="h-4 w-4" />
              <AlertDescription>
                <strong>Browser Storage Info:</strong><br/>
                • Estimated Capacity: {storageInfo.estimatedCapacity}<br/>
                {storageInfo.isQuotaReliable ? (
                  <>• Available: {formatBytes(storageInfo.browserQuota - storageInfo.browserUsed)} of {formatBytes(storageInfo.browserQuota)}</>
                ) : (
                  <>• Browser quota estimate may not be accurate ({formatBytes(storageInfo.browserQuota)} reported)</>
                )}
                <br/>
                <em>Note: Browser storage is cleared when you clear browser data. For permanent storage, consider local server option.</em>
              </AlertDescription>
            </Alert>
          )}
        </>
      )}

      {provider && provider !== 'localstorage' && (
        <Alert>
          <AlertTriangle className="h-4 w-4" />
          <AlertDescription>
            <strong>Setup Required:</strong> Cloud storage providers require backend OAuth configuration. 
            For immediate use, please select Local Storage option.
          </AlertDescription>
        </Alert>
      )}

      {provider && (
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex-1">
                <div className="flex items-center gap-2">
                  {provider === 'localstorage' && <HardDrive className="h-4 w-4" />}
                  <h4 className="font-medium">
                    {provider === 'localstorage' 
                      ? `Setup Local Storage (${storageType === 'browser' ? 'Browser' : 'Server/Network'})` 
                      : `Connect to ${getProviderName(provider)}`
                    }
                  </h4>
                </div>
                <p className="text-sm text-gray-600">
                  {provider === 'localstorage' 
                    ? `Configure ${storageType === 'browser' ? 'browser' : 'local server'} storage for your ESG data`
                    : 'Real cloud integration requires backend setup'
                  }
                </p>
                {connectionStatus === 'connected' && (
                  <p className="text-sm text-green-600 mt-1">
                    ✅ Storage configured and validated successfully
                  </p>
                )}
                {connectionStatus === 'failed' && (
                  <p className="text-sm text-red-600 mt-1">
                    ❌ Storage setup failed - please check configuration
                  </p>
                )}
              </div>
              <Button 
                onClick={() => handleConnection(provider)}
                variant="outline"
                disabled={
                  isConnecting || 
                  !folderName.trim() || 
                  (storageType === 'local-server' && (!localStoragePath || !pathValidation?.isValid))
                }
                className="flex items-center gap-2"
              >
                {getConnectionIcon()}
                {isConnecting ? 'Setting up...' : (provider === 'localstorage' ? 'Setup & Test' : 'Setup Required')}
              </Button>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
};
