
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Cloud, AlertTriangle, HardDrive, CheckCircle } from 'lucide-react';
import { SimpleStorageConnection } from './CloudStorage/SimpleStorageConnection';
import { ProfessionalStorageConnection } from './CloudStorage/ProfessionalStorageConnection';
import { StorageRecommendations } from './CloudStorage/StorageRecommendations';
import { validateStorageConfiguration } from '@/utils/storageValidation';
import type { StorageValidationResult } from '@/utils/storageValidation';

export const CloudStorageSettings = () => {
  const [storageType, setStorageType] = useState('simple');
  const [isConfigured, setIsConfigured] = useState(false);
  const [connectionStatus, setConnectionStatus] = useState<'none' | 'connected' | 'failed'>('none');
  const [storageValidation, setStorageValidation] = useState<StorageValidationResult | null>(null);

  useEffect(() => {
    // Check storage configuration on component mount
    const checkStorageStatus = async () => {
      const validation = await validateStorageConfiguration();
      setStorageValidation(validation);
      setIsConfigured(validation.isConfigured);
      setConnectionStatus(validation.isConfigured ? 'connected' : 'none');
      
      if (validation.isConfigured) {
        setStorageType(validation.storageType === 'professional' ? 'professional' : 'simple');
      }
    };
    
    checkStorageStatus();
  }, []);

  const handleConnectionStatusChange = async (status: 'none' | 'connected' | 'failed') => {
    setConnectionStatus(status);
    
    // Re-validate storage configuration when status changes
    const validation = await validateStorageConfiguration();
    setStorageValidation(validation);
    setIsConfigured(validation.isConfigured);
  };

  const handleConfigSaved = async (configured: boolean) => {
    setIsConfigured(configured);
    
    // Re-validate after saving
    const validation = await validateStorageConfiguration();
    setStorageValidation(validation);
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Cloud className="h-5 w-5" />
            Data Storage Configuration
            {storageValidation?.isConfigured && (
              <CheckCircle className="h-5 w-5 text-green-600" />
            )}
          </CardTitle>
          <p className="text-sm text-gray-600">
            Configure where all your ESG data and reports will be stored - cloud storage or local storage.
          </p>
        </CardHeader>
        <CardContent className="space-y-6">
          {storageValidation && (
            <Alert className={storageValidation.isConfigured ? 'border-green-200 bg-green-50' : 'border-orange-200 bg-orange-50'}>
              <AlertTriangle className="h-4 w-4" />
              <AlertDescription>
                <strong>Storage Status:</strong>{' '}
                {storageValidation.isConfigured ? (
                  <span className="text-green-800">
                    ✅ Configured and functional ({storageValidation.provider} - {storageValidation.storageType})
                  </span>
                ) : (
                  <span className="text-orange-800">
                    ⚠️ Not configured - {storageValidation.errorMessage}
                  </span>
                )}
              </AlertDescription>
            </Alert>
          )}

          <Alert>
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              <strong>Important:</strong> All your ESG data will be stored in your chosen location. 
              You are responsible for maintaining adequate storage capacity and ensuring data security 
              at the storage provider level. <strong>Storage must be configured before you can add or edit any data.</strong>
            </AlertDescription>
          </Alert>

          <Tabs value={storageType} onValueChange={setStorageType} className="w-full">
            <TabsList className="grid w-full grid-cols-2">
              <TabsTrigger value="simple" className="flex items-center gap-2">
                <Cloud className="h-4 w-4" />
                Simple Setup
              </TabsTrigger>
              <TabsTrigger value="professional" className="flex items-center gap-2">
                <HardDrive className="h-4 w-4" />
                Professional Setup
              </TabsTrigger>
            </TabsList>

            <TabsContent value="simple" className="space-y-6">
              <SimpleStorageConnection onConnectionStatusChange={handleConnectionStatusChange} />

              <Alert>
                <AlertDescription>
                  <strong>Simple Setup Benefits:</strong>
                  <ul className="list-disc list-inside mt-2 space-y-1">
                    <li>Easy local storage setup with immediate functionality</li>
                    <li>No technical configuration required</li>
                    <li>Perfect for individual users and small businesses</li>
                    <li>Automatic folder organization and data validation</li>
                    <li>Complete data privacy (stays on your device)</li>
                  </ul>
                </AlertDescription>
              </Alert>
            </TabsContent>

            <TabsContent value="professional" className="space-y-6">
              <ProfessionalStorageConnection onConfigSaved={handleConfigSaved} />

              <Alert>
                <AlertDescription>
                  <strong>Professional Setup Benefits:</strong>
                  <ul className="list-disc list-inside mt-2 space-y-1">
                    <li>Configuration validation and credential testing</li>
                    <li>Support for enterprise cloud storage providers</li>
                    <li>Advanced security configurations and encryption</li>
                    <li>Scalable for large datasets and team environments</li>
                    <li>Custom compliance and backup requirements</li>
                  </ul>
                </AlertDescription>
              </Alert>
            </TabsContent>
          </Tabs>

          <StorageRecommendations />

          {isConfigured && storageValidation?.canWrite && (
            <Alert className="border-green-200 bg-green-50">
              <CheckCircle className="h-4 w-4 text-green-600" />
              <AlertDescription className="text-green-800">
                ✅ <strong>Storage Ready:</strong> Your storage has been configured and tested successfully. 
                All your data will now be securely stored in your designated storage location. 
                You can now add and edit data throughout the application.
              </AlertDescription>
            </Alert>
          )}

          {!isConfigured && (
            <Alert className="border-red-200 bg-red-50">
              <AlertTriangle className="h-4 w-4 text-red-600" />
              <AlertDescription className="text-red-800">
                ⚠️ <strong>Storage Required:</strong> You must configure and test your storage connection 
                before you can add or edit any data in the application. Please complete the setup above.
              </AlertDescription>
            </Alert>
          )}
        </CardContent>
      </Card>
    </div>
  );
};
