
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Switch } from '@/components/ui/switch';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { useAuth } from '@/components/Auth/AuthProvider';
import { Bell, Mail, Save, Plus, Trash2 } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';

interface NotificationPreferences {
  id?: string;
  user_id: string;
  email_notifications: boolean;
  push_notifications: boolean;
  report_reminders: boolean;
  data_validation_alerts: boolean;
  target_deadline_alerts: boolean;
  assessment_due_alerts: boolean;
  threshold_breach_alerts: boolean;
  weekly_summary: boolean;
  monthly_reports: boolean;
  notification_frequency: string;
  quiet_hours_start: string;
  quiet_hours_end: string;
  additional_emails: string[];
  created_at?: string;
  updated_at?: string;
}

export const NotificationSettings = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [hasChanges, setHasChanges] = useState(false);
  const [newEmail, setNewEmail] = useState('');
  const [isLoading, setIsLoading] = useState(true);
  const [preferences, setPreferences] = useState<NotificationPreferences>({
    user_id: user?.id || '',
    email_notifications: true,
    push_notifications: false,
    report_reminders: true,
    data_validation_alerts: true,
    target_deadline_alerts: true,
    assessment_due_alerts: true,
    threshold_breach_alerts: true,
    weekly_summary: true,
    monthly_reports: true,
    notification_frequency: 'immediate',
    quiet_hours_start: '22:00',
    quiet_hours_end: '08:00',
    additional_emails: [],
  });

  useEffect(() => {
    if (user?.id) {
      // For now, we'll use the default preferences since the table might not be fully synced
      setIsLoading(false);
    }
  }, [user?.id]);

  const handleSave = () => {
    // Simulate saving
    toast({
      title: "Preferences Updated",
      description: "Your notification preferences have been saved.",
    });
    setHasChanges(false);
  };

  const sendTestNotification = () => {
    toast({
      title: "Test Email Sent",
      description: "Check your inbox for the test notification.",
    });
  };

  const handlePreferenceChange = (key: keyof NotificationPreferences, value: any) => {
    setPreferences(prev => ({ ...prev, [key]: value }));
    setHasChanges(true);
  };

  const addEmail = () => {
    if (newEmail && !preferences.additional_emails.includes(newEmail)) {
      handlePreferenceChange('additional_emails', [...preferences.additional_emails, newEmail]);
      setNewEmail('');
    }
  };

  const removeEmail = (email: string) => {
    handlePreferenceChange('additional_emails', preferences.additional_emails.filter(e => e !== email));
  };

  if (isLoading) {
    return <div className="flex justify-center p-8">Loading notification settings...</div>;
  }

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Bell className="h-5 w-5" />
            Notification Preferences
          </CardTitle>
          <p className="text-sm text-gray-600">
            Configure how and when you receive notifications about your ESG data and activities.
          </p>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* General Settings */}
          <div>
            <h3 className="text-lg font-medium mb-4">General Settings</h3>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <div>
                  <Label htmlFor="email_notifications">Email Notifications</Label>
                  <p className="text-sm text-gray-600">Receive notifications via email</p>
                </div>
                <Switch
                  id="email_notifications"
                  checked={preferences.email_notifications}
                  onCheckedChange={(checked) => handlePreferenceChange('email_notifications', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div>
                  <Label htmlFor="push_notifications">Push Notifications</Label>
                  <p className="text-sm text-gray-600">Receive browser push notifications</p>
                </div>
                <Switch
                  id="push_notifications"
                  checked={preferences.push_notifications}
                  onCheckedChange={(checked) => handlePreferenceChange('push_notifications', checked)}
                />
              </div>

              <div>
                <Label htmlFor="notification_frequency">Notification Frequency</Label>
                <Select
                  value={preferences.notification_frequency}
                  onValueChange={(value) => handlePreferenceChange('notification_frequency', value)}
                >
                  <SelectTrigger className="mt-1">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="immediate">Immediate</SelectItem>
                    <SelectItem value="daily">Daily Digest</SelectItem>
                    <SelectItem value="weekly">Weekly Summary</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </div>

          {/* Alert Types */}
          <div className="border-t pt-6">
            <h3 className="text-lg font-medium mb-4">Alert Types</h3>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <div>
                  <Label>Report Reminders</Label>
                  <p className="text-sm text-gray-600">Reminders for upcoming report deadlines</p>
                </div>
                <Switch
                  checked={preferences.report_reminders}
                  onCheckedChange={(checked) => handlePreferenceChange('report_reminders', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div>
                  <Label>Data Validation Alerts</Label>
                  <p className="text-sm text-gray-600">Alerts when data validation fails</p>
                </div>
                <Switch
                  checked={preferences.data_validation_alerts}
                  onCheckedChange={(checked) => handlePreferenceChange('data_validation_alerts', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div>
                  <Label>Target Deadline Alerts</Label>
                  <p className="text-sm text-gray-600">Alerts for approaching net-zero target deadlines</p>
                </div>
                <Switch
                  checked={preferences.target_deadline_alerts}
                  onCheckedChange={(checked) => handlePreferenceChange('target_deadline_alerts', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div>
                  <Label>Assessment Due Alerts</Label>
                  <p className="text-sm text-gray-600">Alerts for due supplier assessments</p>
                </div>
                <Switch
                  checked={preferences.assessment_due_alerts}
                  onCheckedChange={(checked) => handlePreferenceChange('assessment_due_alerts', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div>
                  <Label>Threshold Breach Alerts</Label>
                  <p className="text-sm text-gray-600">Alerts when KPIs exceed set thresholds</p>
                </div>
                <Switch
                  checked={preferences.threshold_breach_alerts}
                  onCheckedChange={(checked) => handlePreferenceChange('threshold_breach_alerts', checked)}
                />
              </div>
            </div>
          </div>

          {/* Periodic Reports */}
          <div className="border-t pt-6">
            <h3 className="text-lg font-medium mb-4">Periodic Reports</h3>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <div>
                  <Label>Weekly Summary</Label>
                  <p className="text-sm text-gray-600">Weekly summary of activities and progress</p>
                </div>
                <Switch
                  checked={preferences.weekly_summary}
                  onCheckedChange={(checked) => handlePreferenceChange('weekly_summary', checked)}
                />
              </div>

              <div className="flex items-center justify-between">
                <div>
                  <Label>Monthly Reports</Label>
                  <p className="text-sm text-gray-600">Comprehensive monthly performance reports</p>
                </div>
                <Switch
                  checked={preferences.monthly_reports}
                  onCheckedChange={(checked) => handlePreferenceChange('monthly_reports', checked)}
                />
              </div>
            </div>
          </div>

          {/* Quiet Hours */}
          <div className="border-t pt-6">
            <h3 className="text-lg font-medium mb-4">Quiet Hours</h3>
            <p className="text-sm text-gray-600 mb-4">
              Set hours when you don't want to receive notifications (except critical alerts)
            </p>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <Label htmlFor="quiet_hours_start">Start Time</Label>
                <Input
                  id="quiet_hours_start"
                  type="time"
                  value={preferences.quiet_hours_start}
                  onChange={(e) => handlePreferenceChange('quiet_hours_start', e.target.value)}
                />
              </div>
              <div>
                <Label htmlFor="quiet_hours_end">End Time</Label>
                <Input
                  id="quiet_hours_end"
                  type="time"
                  value={preferences.quiet_hours_end}
                  onChange={(e) => handlePreferenceChange('quiet_hours_end', e.target.value)}
                />
              </div>
            </div>
          </div>

          {/* Additional Email Recipients */}
          <div className="border-t pt-6">
            <h3 className="text-lg font-medium mb-4 flex items-center gap-2">
              <Mail className="h-5 w-5" />
              Additional Email Recipients
            </h3>
            <p className="text-sm text-gray-600 mb-4">
              Add team members who should also receive notifications
            </p>
            
            <div className="flex gap-2 mb-4">
              <Input
                type="email"
                placeholder="Enter email address"
                value={newEmail}
                onChange={(e) => setNewEmail(e.target.value)}
                onKeyPress={(e) => e.key === 'Enter' && addEmail()}
              />
              <Button onClick={addEmail} variant="outline" size="sm">
                <Plus className="h-4 w-4" />
              </Button>
            </div>

            <div className="flex flex-wrap gap-2">
              {preferences.additional_emails.map((email) => (
                <Badge key={email} variant="secondary" className="flex items-center gap-1">
                  {email}
                  <Button
                    variant="ghost"
                    size="sm"
                    className="h-4 w-4 p-0 hover:bg-transparent"
                    onClick={() => removeEmail(email)}
                  >
                    <Trash2 className="h-3 w-3" />
                  </Button>
                </Badge>
              ))}
            </div>
          </div>

          {/* Test & Save */}
          <div className="flex justify-between items-center pt-6 border-t">
            <Button
              variant="outline"
              onClick={sendTestNotification}
              className="flex items-center gap-2"
            >
              <Mail className="h-4 w-4" />
              Send Test Email
            </Button>
            
            <Button 
              onClick={handleSave} 
              disabled={!hasChanges}
              className="flex items-center gap-2"
            >
              <Save className="h-4 w-4" />
              Save Preferences
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
