
import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { supabase } from '@/integrations/supabase/client';
import { useAuth } from '@/components/Auth/AuthProvider';
import { Plus, FileText, Calendar, CheckCircle } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';

interface Assessment {
  id: string;
  supplier_id: string | null;
  assessment_type: string;
  assessment_date: string;
  assessor_name: string | null;
  overall_score: number | null;
  environmental_score: number | null;
  social_score: number | null;
  governance_score: number | null;
  findings: string | null;
  recommendations: string | null;
  action_items: string | null;
  follow_up_date: string | null;
  status: string | null;
  suppliers?: {
    supplier_name: string;
  };
}

interface AssessmentManagerProps {
  onUpdate: () => void;
}

export const AssessmentManager = ({ onUpdate }: AssessmentManagerProps) => {
  const { user } = useAuth();
  const { toast } = useToast();
  const queryClient = useQueryClient();
  const [isAddingAssessment, setIsAddingAssessment] = useState(false);
  const [newAssessment, setNewAssessment] = useState({
    supplier_id: '',
    assessment_type: 'annual',
    assessment_date: new Date().toISOString().split('T')[0],
    assessor_name: '',
    overall_score: '',
    environmental_score: '',
    social_score: '',
    governance_score: '',
    findings: '',
    recommendations: '',
    action_items: '',
    follow_up_date: '',
    status: 'pending'
  });

  const { data: assessments, isLoading } = useQuery({
    queryKey: ['supply-chain-assessments', user?.id],
    queryFn: async () => {
      const { data, error } = await supabase
        .from('supply_chain_assessments')
        .select(`
          *,
          suppliers (
            supplier_name
          )
        `)
        .eq('user_id', user?.id)
        .order('assessment_date', { ascending: false });
      
      if (error) throw error;
      return data as Assessment[];
    },
    enabled: !!user?.id,
  });

  const { data: suppliers } = useQuery({
    queryKey: ['suppliers', user?.id],
    queryFn: async () => {
      const { data, error } = await supabase
        .from('suppliers')
        .select('id, supplier_name')
        .eq('user_id', user?.id)
        .order('supplier_name');
      
      if (error) throw error;
      return data;
    },
    enabled: !!user?.id,
  });

  const createAssessmentMutation = useMutation({
    mutationFn: async (assessment: any) => {
      const { data, error } = await supabase
        .from('supply_chain_assessments')
        .insert([{ ...assessment, user_id: user?.id }])
        .select()
        .single();
      
      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['supply-chain-assessments'] });
      toast({
        title: "Assessment Created",
        description: "Supply chain assessment has been created successfully.",
      });
      setIsAddingAssessment(false);
      setNewAssessment({
        supplier_id: '',
        assessment_type: 'annual',
        assessment_date: new Date().toISOString().split('T')[0],
        assessor_name: '',
        overall_score: '',
        environmental_score: '',
        social_score: '',
        governance_score: '',
        findings: '',
        recommendations: '',
        action_items: '',
        follow_up_date: '',
        status: 'pending'
      });
      onUpdate();
    },
    onError: (error) => {
      toast({
        title: "Error",
        description: "Failed to create assessment. Please try again.",
        variant: "destructive",
      });
    },
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    const assessmentData = {
      ...newAssessment,
      overall_score: newAssessment.overall_score ? parseFloat(newAssessment.overall_score) : null,
      environmental_score: newAssessment.environmental_score ? parseFloat(newAssessment.environmental_score) : null,
      social_score: newAssessment.social_score ? parseFloat(newAssessment.social_score) : null,
      governance_score: newAssessment.governance_score ? parseFloat(newAssessment.governance_score) : null,
      supplier_id: newAssessment.supplier_id || null,
      follow_up_date: newAssessment.follow_up_date || null,
    };
    createAssessmentMutation.mutate(assessmentData);
  };

  const getStatusBadge = (status: string | null) => {
    const statusColors = {
      completed: 'bg-green-100 text-green-800',
      pending: 'bg-yellow-100 text-yellow-800',
      in_progress: 'bg-blue-100 text-blue-800',
    };
    return statusColors[status as keyof typeof statusColors] || 'bg-gray-100 text-gray-800';
  };

  if (isLoading) {
    return <div className="flex justify-center p-8">Loading assessments...</div>;
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold">Supply Chain Assessments</h3>
        <Dialog open={isAddingAssessment} onOpenChange={setIsAddingAssessment}>
          <DialogTrigger asChild>
            <Button className="flex items-center gap-2">
              <Plus className="h-4 w-4" />
              New Assessment
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
            <DialogHeader>
              <DialogTitle>Create New Assessment</DialogTitle>
            </DialogHeader>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="supplier">Supplier</Label>
                  <Select
                    value={newAssessment.supplier_id}
                    onValueChange={(value) => setNewAssessment({ ...newAssessment, supplier_id: value })}
                  >
                    <SelectTrigger>
                      <SelectValue placeholder="Select supplier" />
                    </SelectTrigger>
                    <SelectContent>
                      {suppliers?.map((supplier) => (
                        <SelectItem key={supplier.id} value={supplier.id}>
                          {supplier.supplier_name}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="assessment_type">Assessment Type</Label>
                  <Select
                    value={newAssessment.assessment_type}
                    onValueChange={(value) => setNewAssessment({ ...newAssessment, assessment_type: value })}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="initial">Initial</SelectItem>
                      <SelectItem value="annual">Annual</SelectItem>
                      <SelectItem value="audit">Audit</SelectItem>
                      <SelectItem value="incident_based">Incident Based</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="assessment_date">Assessment Date</Label>
                  <Input
                    id="assessment_date"
                    type="date"
                    value={newAssessment.assessment_date}
                    onChange={(e) => setNewAssessment({ ...newAssessment, assessment_date: e.target.value })}
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="assessor_name">Assessor Name</Label>
                  <Input
                    id="assessor_name"
                    value={newAssessment.assessor_name}
                    onChange={(e) => setNewAssessment({ ...newAssessment, assessor_name: e.target.value })}
                    placeholder="Enter assessor name"
                  />
                </div>

                <div>
                  <Label htmlFor="overall_score">Overall Score (0-100)</Label>
                  <Input
                    id="overall_score"
                    type="number"
                    min="0"
                    max="100"
                    value={newAssessment.overall_score}
                    onChange={(e) => setNewAssessment({ ...newAssessment, overall_score: e.target.value })}
                    placeholder="0-100"
                  />
                </div>

                <div>
                  <Label htmlFor="environmental_score">Environmental Score</Label>
                  <Input
                    id="environmental_score"
                    type="number"
                    min="0"
                    max="100"
                    value={newAssessment.environmental_score}
                    onChange={(e) => setNewAssessment({ ...newAssessment, environmental_score: e.target.value })}
                    placeholder="0-100"
                  />
                </div>

                <div>
                  <Label htmlFor="social_score">Social Score</Label>
                  <Input
                    id="social_score"
                    type="number"
                    min="0"
                    max="100"
                    value={newAssessment.social_score}
                    onChange={(e) => setNewAssessment({ ...newAssessment, social_score: e.target.value })}
                    placeholder="0-100"
                  />
                </div>

                <div>
                  <Label htmlFor="governance_score">Governance Score</Label>
                  <Input
                    id="governance_score"
                    type="number"
                    min="0"
                    max="100"
                    value={newAssessment.governance_score}
                    onChange={(e) => setNewAssessment({ ...newAssessment, governance_score: e.target.value })}
                    placeholder="0-100"
                  />
                </div>
              </div>

              <div>
                <Label htmlFor="findings">Key Findings</Label>
                <Textarea
                  id="findings"
                  value={newAssessment.findings}
                  onChange={(e) => setNewAssessment({ ...newAssessment, findings: e.target.value })}
                  placeholder="Document key findings from the assessment"
                  rows={3}
                />
              </div>

              <div>
                <Label htmlFor="recommendations">Recommendations</Label>
                <Textarea
                  id="recommendations"
                  value={newAssessment.recommendations}
                  onChange={(e) => setNewAssessment({ ...newAssessment, recommendations: e.target.value })}
                  placeholder="Provide recommendations for improvement"
                  rows={3}
                />
              </div>

              <div>
                <Label htmlFor="action_items">Action Items</Label>
                <Textarea
                  id="action_items"
                  value={newAssessment.action_items}
                  onChange={(e) => setNewAssessment({ ...newAssessment, action_items: e.target.value })}
                  placeholder="List specific action items to be completed"
                  rows={3}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="follow_up_date">Follow-up Date</Label>
                  <Input
                    id="follow_up_date"
                    type="date"
                    value={newAssessment.follow_up_date}
                    onChange={(e) => setNewAssessment({ ...newAssessment, follow_up_date: e.target.value })}
                  />
                </div>

                <div>
                  <Label htmlFor="status">Status</Label>
                  <Select
                    value={newAssessment.status}
                    onValueChange={(value) => setNewAssessment({ ...newAssessment, status: value })}
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="pending">Pending</SelectItem>
                      <SelectItem value="in_progress">In Progress</SelectItem>
                      <SelectItem value="completed">Completed</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="flex justify-end gap-2 pt-4">
                <Button type="button" variant="outline" onClick={() => setIsAddingAssessment(false)}>
                  Cancel
                </Button>
                <Button type="submit" disabled={createAssessmentMutation.isPending}>
                  {createAssessmentMutation.isPending ? 'Creating...' : 'Create Assessment'}
                </Button>
              </div>
            </form>
          </DialogContent>
        </Dialog>
      </div>

      <div className="grid gap-4">
        {assessments?.map((assessment) => (
          <Card key={assessment.id}>
            <CardHeader>
              <div className="flex justify-between items-start">
                <div>
                  <CardTitle className="text-lg">
                    {assessment.suppliers?.supplier_name || 'Unknown Supplier'}
                  </CardTitle>
                  <p className="text-sm text-gray-600 capitalize">
                    {assessment.assessment_type.replace('_', ' ')} Assessment
                  </p>
                </div>
                <Badge className={getStatusBadge(assessment.status)}>
                  {assessment.status?.replace('_', ' ') || 'Unknown'}
                </Badge>
              </div>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
                <div className="text-center">
                  <div className="text-2xl font-bold text-blue-600">
                    {assessment.overall_score || 'N/A'}
                  </div>
                  <div className="text-sm text-gray-600">Overall</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-green-600">
                    {assessment.environmental_score || 'N/A'}
                  </div>
                  <div className="text-sm text-gray-600">Environmental</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-purple-600">
                    {assessment.social_score || 'N/A'}
                  </div>
                  <div className="text-sm text-gray-600">Social</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-orange-600">
                    {assessment.governance_score || 'N/A'}
                  </div>
                  <div className="text-sm text-gray-600">Governance</div>
                </div>
              </div>

              <div className="flex items-center gap-4 text-sm text-gray-600 mb-4">
                <div className="flex items-center gap-1">
                  <Calendar className="h-4 w-4" />
                  {new Date(assessment.assessment_date).toLocaleDateString()}
                </div>
                {assessment.assessor_name && (
                  <div className="flex items-center gap-1">
                    <CheckCircle className="h-4 w-4" />
                    {assessment.assessor_name}
                  </div>
                )}
                {assessment.follow_up_date && (
                  <div className="flex items-center gap-1">
                    <FileText className="h-4 w-4" />
                    Follow-up: {new Date(assessment.follow_up_date).toLocaleDateString()}
                  </div>
                )}
              </div>

              {assessment.findings && (
                <div className="mb-3">
                  <h4 className="font-medium mb-1">Key Findings:</h4>
                  <p className="text-sm text-gray-700">{assessment.findings}</p>
                </div>
              )}

              {assessment.recommendations && (
                <div className="mb-3">
                  <h4 className="font-medium mb-1">Recommendations:</h4>
                  <p className="text-sm text-gray-700">{assessment.recommendations}</p>
                </div>
              )}

              {assessment.action_items && (
                <div>
                  <h4 className="font-medium mb-1">Action Items:</h4>
                  <p className="text-sm text-gray-700">{assessment.action_items}</p>
                </div>
              )}
            </CardContent>
          </Card>
        ))}

        {!assessments?.length && (
          <Card>
            <CardContent className="text-center py-8">
              <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No Assessments Yet</h3>
              <p className="text-gray-600 mb-4">
                Start by creating your first supply chain assessment.
              </p>
              <Button onClick={() => setIsAddingAssessment(true)}>
                <Plus className="h-4 w-4 mr-2" />
                Create Assessment
              </Button>
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  );
};
