
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Switch } from '@/components/ui/switch';
import { Plus, Edit, Trash2, Users } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

interface Supplier {
  id: string;
  supplier_name: string;
  supplier_type: string;
  industry_sector: string;
  country: string;
  region: string;
  contact_email: string;
  contact_phone: string;
  annual_spend: number;
  currency: string;
  esg_assessment_score: number;
  carbon_footprint: number;
  environmental_compliance: boolean;
  social_compliance: boolean;
  governance_compliance: boolean;
  risk_level: string;
  status: string;
}

interface SupplierManagerProps {
  onUpdate: () => void;
}

export const SupplierManager: React.FC<SupplierManagerProps> = ({ onUpdate }) => {
  const [suppliers, setSuppliers] = useState<Supplier[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [showForm, setShowForm] = useState(false);
  const [editingSupplier, setEditingSupplier] = useState<Supplier | null>(null);
  const [formData, setFormData] = useState({
    supplier_name: '',
    supplier_type: 'tier_1',
    industry_sector: '',
    country: '',
    region: '',
    contact_email: '',
    contact_phone: '',
    annual_spend: '',
    currency: 'USD',
    esg_assessment_score: '',
    carbon_footprint: '',
    environmental_compliance: false,
    social_compliance: false,
    governance_compliance: false,
    risk_level: 'medium',
    status: 'active'
  });
  const { toast } = useToast();

  useEffect(() => {
    fetchSuppliers();
  }, []);

  const fetchSuppliers = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from('suppliers')
        .select('*')
        .eq('user_id', user.id)
        .order('created_at', { ascending: false });

      if (error) throw error;
      setSuppliers(data || []);
    } catch (error) {
      console.error('Error fetching suppliers:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('User not authenticated');

      const supplierData = {
        user_id: user.id,
        supplier_name: formData.supplier_name,
        supplier_type: formData.supplier_type,
        industry_sector: formData.industry_sector,
        country: formData.country,
        region: formData.region,
        contact_email: formData.contact_email,
        contact_phone: formData.contact_phone,
        annual_spend: formData.annual_spend ? parseFloat(formData.annual_spend) : null,
        currency: formData.currency,
        esg_assessment_score: formData.esg_assessment_score ? parseFloat(formData.esg_assessment_score) : null,
        carbon_footprint: formData.carbon_footprint ? parseFloat(formData.carbon_footprint) : null,
        environmental_compliance: formData.environmental_compliance,
        social_compliance: formData.social_compliance,
        governance_compliance: formData.governance_compliance,
        risk_level: formData.risk_level,
        status: formData.status
      };

      if (editingSupplier) {
        const { error } = await supabase
          .from('suppliers')
          .update(supplierData)
          .eq('id', editingSupplier.id);
        
        if (error) throw error;
        toast({ title: "Supplier updated successfully" });
      } else {
        const { error } = await supabase
          .from('suppliers')
          .insert(supplierData);
        
        if (error) throw error;
        toast({ title: "Supplier added successfully" });
      }

      resetForm();
      fetchSuppliers();
      onUpdate();
    } catch (error) {
      console.error('Error saving supplier:', error);
      toast({
        title: "Error saving supplier",
        description: "Please try again",
        variant: "destructive"
      });
    }
  };

  const resetForm = () => {
    setFormData({
      supplier_name: '',
      supplier_type: 'tier_1',
      industry_sector: '',
      country: '',
      region: '',
      contact_email: '',
      contact_phone: '',
      annual_spend: '',
      currency: 'USD',
      esg_assessment_score: '',
      carbon_footprint: '',
      environmental_compliance: false,
      social_compliance: false,
      governance_compliance: false,
      risk_level: 'medium',
      status: 'active'
    });
    setEditingSupplier(null);
    setShowForm(false);
  };

  const handleEdit = (supplier: Supplier) => {
    setFormData({
      supplier_name: supplier.supplier_name,
      supplier_type: supplier.supplier_type,
      industry_sector: supplier.industry_sector || '',
      country: supplier.country || '',
      region: supplier.region || '',
      contact_email: supplier.contact_email || '',
      contact_phone: supplier.contact_phone || '',
      annual_spend: supplier.annual_spend?.toString() || '',
      currency: supplier.currency,
      esg_assessment_score: supplier.esg_assessment_score?.toString() || '',
      carbon_footprint: supplier.carbon_footprint?.toString() || '',
      environmental_compliance: supplier.environmental_compliance,
      social_compliance: supplier.social_compliance,
      governance_compliance: supplier.governance_compliance,
      risk_level: supplier.risk_level,
      status: supplier.status
    });
    setEditingSupplier(supplier);
    setShowForm(true);
  };

  const handleDelete = async (id: string) => {
    try {
      const { error } = await supabase
        .from('suppliers')
        .delete()
        .eq('id', id);

      if (error) throw error;
      
      toast({ title: "Supplier deleted successfully" });
      fetchSuppliers();
      onUpdate();
    } catch (error) {
      console.error('Error deleting supplier:', error);
      toast({
        title: "Error deleting supplier",
        variant: "destructive"
      });
    }
  };

  const getRiskColor = (risk: string) => {
    switch (risk) {
      case 'high': return 'bg-red-100 text-red-800';
      case 'medium': return 'bg-yellow-100 text-yellow-800';
      case 'low': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-100 text-green-800';
      case 'under_review': return 'bg-yellow-100 text-yellow-800';
      case 'suspended': return 'bg-red-100 text-red-800';
      case 'terminated': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  if (isLoading) {
    return <div className="animate-pulse h-64 bg-gray-200 rounded"></div>;
  }

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold">Supplier Management</h3>
        <Button onClick={() => setShowForm(true)} className="flex items-center gap-2">
          <Plus className="h-4 w-4" />
          Add Supplier
        </Button>
      </div>

      {showForm && (
        <Card>
          <CardHeader>
            <CardTitle>
              {editingSupplier ? 'Edit Supplier' : 'Add New Supplier'}
            </CardTitle>
          </CardHeader>
          <CardContent>
            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <Label>Supplier Name *</Label>
                  <Input
                    value={formData.supplier_name}
                    onChange={(e) => setFormData({...formData, supplier_name: e.target.value})}
                    required
                  />
                </div>
                <div>
                  <Label>Supplier Type *</Label>
                  <Select value={formData.supplier_type} onValueChange={(value) => setFormData({...formData, supplier_type: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="tier_1">Tier 1</SelectItem>
                      <SelectItem value="tier_2">Tier 2</SelectItem>
                      <SelectItem value="tier_3">Tier 3</SelectItem>
                      <SelectItem value="critical">Critical</SelectItem>
                      <SelectItem value="non_critical">Non-Critical</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>Industry Sector</Label>
                  <Input
                    value={formData.industry_sector}
                    onChange={(e) => setFormData({...formData, industry_sector: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Country</Label>
                  <Input
                    value={formData.country}
                    onChange={(e) => setFormData({...formData, country: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Region</Label>
                  <Input
                    value={formData.region}
                    onChange={(e) => setFormData({...formData, region: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Contact Email</Label>
                  <Input
                    type="email"
                    value={formData.contact_email}
                    onChange={(e) => setFormData({...formData, contact_email: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Contact Phone</Label>
                  <Input
                    value={formData.contact_phone}
                    onChange={(e) => setFormData({...formData, contact_phone: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Annual Spend</Label>
                  <Input
                    type="number"
                    value={formData.annual_spend}
                    onChange={(e) => setFormData({...formData, annual_spend: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Currency</Label>
                  <Select value={formData.currency} onValueChange={(value) => setFormData({...formData, currency: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="USD">USD</SelectItem>
                      <SelectItem value="EUR">EUR</SelectItem>
                      <SelectItem value="GBP">GBP</SelectItem>
                      <SelectItem value="JPY">JPY</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label>ESG Assessment Score (0-100)</Label>
                  <Input
                    type="number"
                    min="0"
                    max="100"
                    value={formData.esg_assessment_score}
                    onChange={(e) => setFormData({...formData, esg_assessment_score: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Carbon Footprint (tCO2e)</Label>
                  <Input
                    type="number"
                    value={formData.carbon_footprint}
                    onChange={(e) => setFormData({...formData, carbon_footprint: e.target.value})}
                  />
                </div>
                <div>
                  <Label>Risk Level</Label>
                  <Select value={formData.risk_level} onValueChange={(value) => setFormData({...formData, risk_level: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="high">High</SelectItem>
                      <SelectItem value="medium">Medium</SelectItem>
                      <SelectItem value="low">Low</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div>
                <Label>Status</Label>
                <Select value={formData.status} onValueChange={(value) => setFormData({...formData, status: value})}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="active">Active</SelectItem>
                    <SelectItem value="under_review">Under Review</SelectItem>
                    <SelectItem value="suspended">Suspended</SelectItem>
                    <SelectItem value="terminated">Terminated</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-4">
                <Label>Compliance Status</Label>
                <div className="flex items-center space-x-4">
                  <div className="flex items-center space-x-2">
                    <Switch
                      checked={formData.environmental_compliance}
                      onCheckedChange={(checked) => setFormData({...formData, environmental_compliance: checked})}
                    />
                    <Label>Environmental</Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Switch
                      checked={formData.social_compliance}
                      onCheckedChange={(checked) => setFormData({...formData, social_compliance: checked})}
                    />
                    <Label>Social</Label>
                  </div>
                  <div className="flex items-center space-x-2">
                    <Switch
                      checked={formData.governance_compliance}
                      onCheckedChange={(checked) => setFormData({...formData, governance_compliance: checked})}
                    />
                    <Label>Governance</Label>
                  </div>
                </div>
              </div>

              <div className="flex gap-2">
                <Button type="submit">
                  {editingSupplier ? 'Update Supplier' : 'Add Supplier'}
                </Button>
                <Button type="button" variant="outline" onClick={resetForm}>
                  Cancel
                </Button>
              </div>
            </form>
          </CardContent>
        </Card>
      )}

      <div className="grid grid-cols-1 gap-4">
        {suppliers.map((supplier) => (
          <Card key={supplier.id}>
            <CardContent className="p-4">
              <div className="flex justify-between items-start">
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-2">
                    <h4 className="font-semibold">{supplier.supplier_name}</h4>
                    <Badge className={getStatusColor(supplier.status)}>
                      {supplier.status}
                    </Badge>
                    <Badge className={getRiskColor(supplier.risk_level)}>
                      {supplier.risk_level} risk
                    </Badge>
                  </div>
                  
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm mb-3">
                    <div>
                      <p className="text-gray-600">Type</p>
                      <p className="font-medium">{supplier.supplier_type.replace('_', ' ')}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Industry</p>
                      <p className="font-medium">{supplier.industry_sector || 'N/A'}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">Location</p>
                      <p className="font-medium">{supplier.country || 'N/A'}</p>
                    </div>
                    <div>
                      <p className="text-gray-600">ESG Score</p>
                      <p className="font-medium">{supplier.esg_assessment_score || 'N/A'}</p>
                    </div>
                  </div>

                  <div className="flex items-center gap-4 text-sm">
                    <div className="flex items-center gap-1">
                      <div className={`w-2 h-2 rounded-full ${supplier.environmental_compliance ? 'bg-green-500' : 'bg-red-500'}`}></div>
                      <span>Environmental</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <div className={`w-2 h-2 rounded-full ${supplier.social_compliance ? 'bg-green-500' : 'bg-red-500'}`}></div>
                      <span>Social</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <div className={`w-2 h-2 rounded-full ${supplier.governance_compliance ? 'bg-green-500' : 'bg-red-500'}`}></div>
                      <span>Governance</span>
                    </div>
                  </div>
                </div>
                
                <div className="flex gap-2">
                  <Button size="sm" variant="outline" onClick={() => handleEdit(supplier)}>
                    <Edit className="h-4 w-4" />
                  </Button>
                  <Button size="sm" variant="outline" onClick={() => handleDelete(supplier.id)}>
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {suppliers.length === 0 && (
        <Card>
          <CardContent className="p-8 text-center">
            <Users className="h-12 w-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-600 mb-2">No suppliers yet</h3>
            <p className="text-gray-500 mb-4">Start building your supplier database by adding your first supplier.</p>
            <Button onClick={() => setShowForm(true)}>
              Add First Supplier
            </Button>
          </CardContent>
        </Card>
      )}
    </div>
  );
};
