
import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { useQuery } from '@tanstack/react-query';
import { supabase } from '@/integrations/supabase/client';
import { useAuth } from '@/components/Auth/AuthProvider';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart, Pie, Cell, LineChart, Line } from 'recharts';
import { TrendingUp, TrendingDown, AlertTriangle, CheckCircle, Users, Globe } from 'lucide-react';

const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042', '#8884D8'];

export const SupplyChainAnalytics = () => {
  const { user } = useAuth();

  const { data: suppliers, isLoading: suppliersLoading } = useQuery({
    queryKey: ['suppliers-analytics', user?.id],
    queryFn: async () => {
      const { data, error } = await supabase
        .from('suppliers')
        .select('*')
        .eq('user_id', user?.id);
      
      if (error) throw error;
      return data;
    },
    enabled: !!user?.id,
  });

  const { data: assessments, isLoading: assessmentsLoading } = useQuery({
    queryKey: ['assessments-analytics', user?.id],
    queryFn: async () => {
      const { data, error } = await supabase
        .from('supply_chain_assessments')
        .select(`
          *,
          suppliers (
            supplier_name,
            supplier_type,
            region
          )
        `)
        .eq('user_id', user?.id);
      
      if (error) throw error;
      return data;
    },
    enabled: !!user?.id,
  });

  if (suppliersLoading || assessmentsLoading) {
    return <div className="flex justify-center p-8">Loading analytics...</div>;
  }

  // Calculate key metrics
  const totalSuppliers = suppliers?.length || 0;
  const activeSuppliers = suppliers?.filter(s => s.status === 'active').length || 0;
  const avgESGScore = suppliers?.length 
    ? suppliers.reduce((sum, s) => sum + (s.esg_assessment_score || 0), 0) / suppliers.length
    : 0;
  
  const highRiskSuppliers = suppliers?.filter(s => s.risk_level === 'high').length || 0;
  const compliantSuppliers = suppliers?.filter(s => 
    s.environmental_compliance && s.social_compliance && s.governance_compliance
  ).length || 0;

  // Prepare chart data
  const riskDistribution = [
    { name: 'Low Risk', value: suppliers?.filter(s => s.risk_level === 'low').length || 0, color: '#00C49F' },
    { name: 'Medium Risk', value: suppliers?.filter(s => s.risk_level === 'medium').length || 0, color: '#FFBB28' },
    { name: 'High Risk', value: suppliers?.filter(s => s.risk_level === 'high').length || 0, color: '#FF8042' },
  ];

  const regionalDistribution = suppliers?.reduce((acc: any[], supplier) => {
    const region = supplier.region || 'Unknown';
    const existing = acc.find(item => item.name === region);
    if (existing) {
      existing.value += 1;
    } else {
      acc.push({ name: region, value: 1 });
    }
    return acc;
  }, []) || [];

  const esgScoreDistribution = [
    { range: '0-20', count: suppliers?.filter(s => (s.esg_assessment_score || 0) <= 20).length || 0 },
    { range: '21-40', count: suppliers?.filter(s => (s.esg_assessment_score || 0) > 20 && (s.esg_assessment_score || 0) <= 40).length || 0 },
    { range: '41-60', count: suppliers?.filter(s => (s.esg_assessment_score || 0) > 40 && (s.esg_assessment_score || 0) <= 60).length || 0 },
    { range: '61-80', count: suppliers?.filter(s => (s.esg_assessment_score || 0) > 60 && (s.esg_assessment_score || 0) <= 80).length || 0 },
    { range: '81-100', count: suppliers?.filter(s => (s.esg_assessment_score || 0) > 80).length || 0 },
  ];

  const complianceData = [
    { type: 'Environmental', compliant: suppliers?.filter(s => s.environmental_compliance).length || 0, total: totalSuppliers },
    { type: 'Social', compliant: suppliers?.filter(s => s.social_compliance).length || 0, total: totalSuppliers },
    { type: 'Governance', compliant: suppliers?.filter(s => s.governance_compliance).length || 0, total: totalSuppliers },
  ];

  return (
    <div className="space-y-6">
      {/* Key Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Total Suppliers</p>
                <p className="text-3xl font-bold">{totalSuppliers}</p>
                <p className="text-sm text-green-600 flex items-center mt-1">
                  <CheckCircle className="h-4 w-4 mr-1" />
                  {activeSuppliers} active
                </p>
              </div>
              <Users className="h-8 w-8 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Avg ESG Score</p>
                <p className="text-3xl font-bold">{avgESGScore.toFixed(1)}</p>
                <p className="text-sm text-gray-600 mt-1">Out of 100</p>
              </div>
              <TrendingUp className="h-8 w-8 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">High Risk</p>
                <p className="text-3xl font-bold text-red-600">{highRiskSuppliers}</p>
                <p className="text-sm text-gray-600 mt-1">Suppliers</p>
              </div>
              <AlertTriangle className="h-8 w-8 text-red-600" />
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Fully Compliant</p>
                <p className="text-3xl font-bold text-green-600">{compliantSuppliers}</p>
                <p className="text-sm text-gray-600 mt-1">Suppliers</p>
              </div>
              <CheckCircle className="h-8 w-8 text-green-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Charts */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Risk Distribution */}
        <Card>
          <CardHeader>
            <CardTitle>Risk Level Distribution</CardTitle>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <PieChart>
                <Pie
                  data={riskDistribution}
                  cx="50%"
                  cy="50%"
                  outerRadius={80}
                  fill="#8884d8"
                  dataKey="value"
                  label={({ name, value }) => `${name}: ${value}`}
                >
                  {riskDistribution.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={entry.color} />
                  ))}
                </Pie>
                <Tooltip />
              </PieChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        {/* Regional Distribution */}
        <Card>
          <CardHeader>
            <CardTitle>Regional Distribution</CardTitle>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={regionalDistribution}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="name" />
                <YAxis />
                <Tooltip />
                <Bar dataKey="value" fill="#8884d8" />
              </BarChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        {/* ESG Score Distribution */}
        <Card>
          <CardHeader>
            <CardTitle>ESG Score Distribution</CardTitle>
          </CardHeader>
          <CardContent>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={esgScoreDistribution}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="range" />
                <YAxis />
                <Tooltip />
                <Bar dataKey="count" fill="#00C49F" />
              </BarChart>
            </ResponsiveContainer>
          </CardContent>
        </Card>

        {/* Compliance Analysis */}
        <Card>
          <CardHeader>
            <CardTitle>Compliance Analysis</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {complianceData.map((item) => (
                <div key={item.type} className="space-y-2">
                  <div className="flex justify-between items-center">
                    <span className="text-sm font-medium">{item.type}</span>
                    <span className="text-sm text-gray-600">
                      {item.compliant}/{item.total} ({item.total > 0 ? Math.round((item.compliant / item.total) * 100) : 0}%)
                    </span>
                  </div>
                  <div className="w-full bg-gray-200 rounded-full h-2">
                    <div
                      className="bg-green-600 h-2 rounded-full"
                      style={{
                        width: `${item.total > 0 ? (item.compliant / item.total) * 100 : 0}%`
                      }}
                    ></div>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Recent Assessments Summary */}
      <Card>
        <CardHeader>
          <CardTitle>Assessment Summary</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="text-center">
              <div className="text-2xl font-bold text-blue-600">
                {assessments?.length || 0}
              </div>
              <div className="text-sm text-gray-600">Total Assessments</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">
                {assessments?.filter(a => a.status === 'completed').length || 0}
              </div>
              <div className="text-sm text-gray-600">Completed</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-yellow-600">
                {assessments?.filter(a => a.status === 'pending').length || 0}
              </div>
              <div className="text-sm text-gray-600">Pending</div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Top Performing Suppliers */}
      {suppliers && suppliers.length > 0 && (
        <Card>
          <CardHeader>
            <CardTitle>Top Performing Suppliers</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {suppliers
                .filter(s => s.esg_assessment_score)
                .sort((a, b) => (b.esg_assessment_score || 0) - (a.esg_assessment_score || 0))
                .slice(0, 5)
                .map((supplier) => (
                  <div key={supplier.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <div>
                      <h4 className="font-medium">{supplier.supplier_name}</h4>
                      <p className="text-sm text-gray-600">{supplier.industry_sector}</p>
                    </div>
                    <div className="text-right">
                      <div className="text-lg font-bold text-green-600">
                        {supplier.esg_assessment_score}
                      </div>
                      <div className="text-sm text-gray-600">ESG Score</div>
                    </div>
                  </div>
                ))}
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
};
