
import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Users, AlertTriangle, CheckCircle, TrendingUp, DollarSign, Leaf } from 'lucide-react';
import { supabase } from '@/integrations/supabase/client';

interface SupplyChainSummary {
  totalSuppliers: number;
  activeSuppliers: number;
  criticalSuppliers: number;
  highRiskSuppliers: number;
  totalSpend: number;
  currency: string;
  avgESGScore: number;
  complianceRate: number;
  assessmentsDue: number;
  carbonFootprint: number;
}

export const SupplyChainOverview: React.FC = () => {
  const [summary, setSummary] = useState<SupplyChainSummary>({
    totalSuppliers: 0,
    activeSuppliers: 0,
    criticalSuppliers: 0,
    highRiskSuppliers: 0,
    totalSpend: 0,
    currency: 'USD',
    avgESGScore: 0,
    complianceRate: 0,
    assessmentsDue: 0,
    carbonFootprint: 0
  });
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    fetchSupplyChainSummary();
  }, []);

  const fetchSupplyChainSummary = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data: suppliers, error } = await supabase
        .from('suppliers')
        .select('*')
        .eq('user_id', user.id);

      if (error) throw error;

      if (suppliers && suppliers.length > 0) {
        const totalSuppliers = suppliers.length;
        const activeSuppliers = suppliers.filter(s => s.status === 'active').length;
        const criticalSuppliers = suppliers.filter(s => s.supplier_type === 'critical').length;
        const highRiskSuppliers = suppliers.filter(s => s.risk_level === 'high').length;
        const totalSpend = suppliers.reduce((sum, s) => sum + (Number(s.annual_spend) || 0), 0);
        const avgESGScore = suppliers.reduce((sum, s) => sum + (Number(s.esg_assessment_score) || 0), 0) / totalSuppliers;
        const compliantSuppliers = suppliers.filter(s => 
          s.environmental_compliance && s.social_compliance && s.governance_compliance
        ).length;
        const complianceRate = (compliantSuppliers / totalSuppliers) * 100;
        const carbonFootprint = suppliers.reduce((sum, s) => sum + (Number(s.carbon_footprint) || 0), 0);
        
        // Calculate assessments due (next assessment date is in the past or within 30 days)
        const thirtyDaysFromNow = new Date();
        thirtyDaysFromNow.setDate(thirtyDaysFromNow.getDate() + 30);
        const assessmentsDue = suppliers.filter(s => {
          if (!s.next_assessment_date) return true;
          return new Date(s.next_assessment_date) <= thirtyDaysFromNow;
        }).length;

        setSummary({
          totalSuppliers,
          activeSuppliers,
          criticalSuppliers,
          highRiskSuppliers,
          totalSpend,
          currency: suppliers[0]?.currency || 'USD',
          avgESGScore,
          complianceRate,
          assessmentsDue,
          carbonFootprint
        });
      }
    } catch (error) {
      console.error('Error fetching supply chain summary:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const getComplianceColor = (rate: number) => {
    if (rate >= 80) return 'text-green-600';
    if (rate >= 60) return 'text-yellow-600';
    return 'text-red-600';
  };

  const getESGColor = (score: number) => {
    if (score >= 80) return 'text-green-600';
    if (score >= 60) return 'text-yellow-600';
    return 'text-red-600';
  };

  if (isLoading) {
    return <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
      {[1, 2, 3, 4, 5, 6].map(i => (
        <Card key={i} className="animate-pulse">
          <div className="h-24 bg-gray-200 rounded"></div>
        </Card>
      ))}
    </div>;
  }

  return (
    <div className="space-y-6">
      {/* Key Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card className="bg-blue-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Users className="h-5 w-5 text-blue-600" />
              <div>
                <p className="text-sm text-gray-600">Total Suppliers</p>
                <p className="text-2xl font-bold">{summary.totalSuppliers}</p>
                <p className="text-sm text-gray-500">{summary.activeSuppliers} active</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-green-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <DollarSign className="h-5 w-5 text-green-600" />
              <div>
                <p className="text-sm text-gray-600">Annual Spend</p>
                <p className="text-2xl font-bold">{summary.totalSpend.toLocaleString()}</p>
                <p className="text-sm text-gray-500">{summary.currency}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-orange-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <AlertTriangle className="h-5 w-5 text-orange-600" />
              <div>
                <p className="text-sm text-gray-600">High Risk</p>
                <p className="text-2xl font-bold">{summary.highRiskSuppliers}</p>
                <p className="text-sm text-gray-500">Critical: {summary.criticalSuppliers}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-purple-50">
          <CardContent className="p-4">
            <div className="flex items-center gap-2">
              <Leaf className="h-5 w-5 text-purple-600" />
              <div>
                <p className="text-sm text-gray-600">Carbon Footprint</p>
                <p className="text-2xl font-bold">{summary.carbonFootprint.toFixed(1)}</p>
                <p className="text-sm text-gray-500">tCO2e annually</p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Performance Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>ESG Performance</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div>
                <div className="flex justify-between items-center mb-2">
                  <span className="text-sm font-medium">Average ESG Score</span>
                  <span className={`text-sm font-bold ${getESGColor(summary.avgESGScore)}`}>
                    {summary.avgESGScore.toFixed(1)}
                  </span>
                </div>
                <Progress value={summary.avgESGScore} className="h-2" />
              </div>
              
              <div>
                <div className="flex justify-between items-center mb-2">
                  <span className="text-sm font-medium">Compliance Rate</span>
                  <span className={`text-sm font-bold ${getComplianceColor(summary.complianceRate)}`}>
                    {summary.complianceRate.toFixed(1)}%
                  </span>
                </div>
                <Progress value={summary.complianceRate} className="h-2" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Risk Assessment</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              <div className="flex justify-between items-center">
                <span className="flex items-center gap-2">
                  <div className="w-3 h-3 bg-red-500 rounded-full"></div>
                  High Risk
                </span>
                <Badge variant="outline">{summary.highRiskSuppliers}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="flex items-center gap-2">
                  <div className="w-3 h-3 bg-orange-500 rounded-full"></div>
                  Critical Suppliers
                </span>
                <Badge variant="outline">{summary.criticalSuppliers}</Badge>
              </div>
              <div className="flex justify-between items-center">
                <span className="flex items-center gap-2">
                  <div className="w-3 h-3 bg-yellow-500 rounded-full"></div>
                  Assessments Due
                </span>
                <Badge variant="outline">{summary.assessmentsDue}</Badge>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Alerts and Recommendations */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <CheckCircle className="h-5 w-5" />
            Key Insights & Recommendations
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            {summary.highRiskSuppliers > 0 && (
              <div className="p-3 bg-red-50 rounded-lg">
                <p className="text-sm text-red-800">
                  <strong>High Risk Alert:</strong> {summary.highRiskSuppliers} suppliers are classified as high risk. 
                  Consider immediate assessment and mitigation strategies.
                </p>
              </div>
            )}
            
            {summary.assessmentsDue > 0 && (
              <div className="p-3 bg-orange-50 rounded-lg">
                <p className="text-sm text-orange-800">
                  <strong>Assessments Due:</strong> {summary.assessmentsDue} suppliers require assessment within 30 days. 
                  Schedule reviews to maintain compliance monitoring.
                </p>
              </div>
            )}
            
            {summary.complianceRate < 80 && (
              <div className="p-3 bg-yellow-50 rounded-lg">
                <p className="text-sm text-yellow-800">
                  <strong>Compliance Gap:</strong> Current compliance rate is {summary.complianceRate.toFixed(1)}%. 
                  Focus on improving environmental, social, and governance standards.
                </p>
              </div>
            )}
            
            {summary.avgESGScore >= 80 && (
              <div className="p-3 bg-green-50 rounded-lg">
                <p className="text-sm text-green-800">
                  <strong>Strong Performance:</strong> Your supply chain demonstrates excellent ESG performance 
                  with an average score of {summary.avgESGScore.toFixed(1)}. Continue monitoring and improvement initiatives.
                </p>
              </div>
            )}
            
            {summary.totalSuppliers === 0 && (
              <div className="p-3 bg-blue-50 rounded-lg">
                <p className="text-sm text-blue-800">
                  <strong>Getting Started:</strong> Begin by adding your key suppliers and conducting initial 
                  ESG assessments to establish baseline performance metrics.
                </p>
              </div>
            )}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
