
export interface SustainabilityMetric {
  id: string;
  code: string;
  name: string;
  category: string;
  subcategory: string;
  description: string;
  unit: string;
  standardCodes: {
    [key: string]: string[];
  };
  scope?: string;
  targetType?: string;
  dataType: 'number' | 'percentage' | 'boolean' | 'text';
  required?: boolean;
  difficulty: string;
  calculationMethod: string;
  dataSource: string[];
  frequency: string;
  mandatory: boolean;
  standards: Array<{
    name: string;
    code: string;
    requirement: string;
  }>;
  factors: Array<{
    name: string;
    value: number;
    unit: string;
    source: string;
  }>;
  materials?: string[];
  industry: string[];
}

export const sustainabilityMetrics: SustainabilityMetric[] = [
  // Revenue & Employment
  {
    id: 'REVER001',
    code: 'REVER001',
    name: 'Revenue Total',
    category: 'Financial',
    subcategory: 'Economic Performance',
    description: 'Total revenue generated by the organization',
    unit: 'Currency',
    standardCodes: { GRI: ['201-1'], SASB: ['FB-RN-000.A'], SDG: ['8.1', '8.2'] },
    dataType: 'number',
    difficulty: 'Easy',
    calculationMethod: 'Sum of all revenue streams including sales, services, and other income sources',
    dataSource: ['Financial Statements', 'Accounting Records', 'Annual Reports'],
    frequency: 'Annual',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '201-1', requirement: 'Direct economic value generated and distributed' },
      { name: 'SASB', code: 'FB-RN-000.A', requirement: 'Total revenue from food and beverage products' },
      { name: 'SDG', code: '8.1', requirement: 'Sustain per capita economic growth' }
    ],
    factors: [
      { name: 'Revenue Growth Rate', value: 0.05, unit: 'percentage', source: 'Industry Average' }
    ],
    industry: ['All Industries', 'Manufacturing', 'Services', 'Technology']
  },
  {
    id: 'EMPLR001',
    code: 'EMPLR001',
    name: 'Employees Total',
    category: 'Social',
    subcategory: 'Employment',
    description: 'Total number of employees in the organization',
    unit: 'Count',
    standardCodes: { GRI: ['102-7', '401-1'], SASB: ['HC-MS-000.A'], SDG: ['8.5', '8.6'] },
    dataType: 'number',
    difficulty: 'Easy',
    calculationMethod: 'Count of all full-time, part-time, temporary, and contract employees',
    dataSource: ['HR Records', 'Payroll System', 'Employment Database'],
    frequency: 'Annual',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '102-7', requirement: 'Scale of the organization' },
      { name: 'GRI', code: '401-1', requirement: 'New employee hires and employee turnover' },
      { name: 'SASB', code: 'HC-MS-000.A', requirement: 'Number of employees' }
    ],
    factors: [],
    industry: ['All Industries', 'Manufacturing', 'Services', 'Healthcare', 'Technology']
  },

  // Energy Management
  {
    id: 'ENERS001',
    code: 'ENERS001',
    name: 'Energy Reduction Strategy',
    category: 'Environmental',
    subcategory: 'Energy Management',
    description: 'Strategy for reducing energy consumption',
    unit: 'Boolean',
    standardCodes: { GRI: ['302-4'], TCFD: ['Strategy-1'], SDG: ['7.3', '13.2'] },
    dataType: 'boolean',
    difficulty: 'Medium',
    calculationMethod: 'Qualitative assessment of energy reduction initiatives and policies',
    dataSource: ['Energy Management Plans', 'Sustainability Reports', 'Policy Documents'],
    frequency: 'Annual',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '302-4', requirement: 'Reduction of energy consumption' },
      { name: 'TCFD', code: 'Strategy-1', requirement: 'Climate-related risks and opportunities' },
      { name: 'SDG', code: '7.3', requirement: 'Double the global rate of improvement in energy efficiency' }
    ],
    factors: [],
    industry: ['Manufacturing', 'Energy', 'Construction', 'Transportation']
  },
  {
    id: 'ENERS002',
    code: 'ENERS002',
    name: 'Renewable Energy Use Strategy',
    category: 'Environmental',
    subcategory: 'Energy Management',
    description: 'Strategy for increasing renewable energy use',
    unit: 'Boolean',
    standardCodes: { GRI: ['302-1'], TCFD: ['Strategy-2'], SDG: ['7.2', '13.2'] },
    dataType: 'boolean',
    difficulty: 'Medium',
    calculationMethod: 'Assessment of renewable energy procurement and generation strategies',
    dataSource: ['Energy Procurement Contracts', 'Renewable Energy Certificates', 'Energy Plans'],
    frequency: 'Annual',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '302-1', requirement: 'Energy consumption within the organization' },
      { name: 'TCFD', code: 'Strategy-2', requirement: 'Impact of climate-related risks and opportunities' },
      { name: 'SDG', code: '7.2', requirement: 'Increase substantially the share of renewable energy' }
    ],
    factors: [],
    industry: ['All Industries', 'Manufacturing', 'Energy', 'Technology']
  },
  {
    id: 'ENERT001',
    code: 'ENERT001',
    name: 'Energy Reduction Target',
    category: 'Environmental',
    subcategory: 'Energy Management',
    description: 'Target for energy reduction',
    unit: 'Percentage',
    standardCodes: { GRI: ['302-4'], SBTi: ['Target-1'], SDG: ['7.3'] },
    dataType: 'percentage',
    difficulty: 'Medium',
    calculationMethod: 'Percentage reduction target from baseline year energy consumption',
    dataSource: ['Sustainability Targets', 'Energy Management Plans', 'Corporate Commitments'],
    frequency: 'Annual',
    mandatory: false,
    standards: [
      { name: 'GRI', code: '302-4', requirement: 'Reduction of energy consumption' },
      { name: 'SBTi', code: 'Target-1', requirement: 'Science-based emission reduction targets' },
      { name: 'SDG', code: '7.3', requirement: 'Energy efficiency improvement targets' }
    ],
    factors: [
      { name: 'Industry Average Reduction', value: 0.02, unit: 'percentage annually', source: 'IEA' }
    ],
    industry: ['Manufacturing', 'Energy', 'Construction', 'Transportation']
  },
  {
    id: 'ENERT002',
    code: 'ENERT002',
    name: 'Renewable Energy Use Target',
    category: 'Environmental',
    subcategory: 'Energy Management',
    description: 'Target for renewable energy use',
    unit: 'Percentage',
    standardCodes: { GRI: ['302-1'], RE100: ['Target-1'], SDG: ['7.2'] },
    dataType: 'percentage',
    difficulty: 'Medium',
    calculationMethod: 'Percentage of total energy consumption from renewable sources target',
    dataSource: ['Renewable Energy Commitments', 'RE100 Commitments', 'Sustainability Plans'],
    frequency: 'Annual',
    mandatory: false,
    standards: [
      { name: 'GRI', code: '302-1', requirement: 'Energy consumption within the organization' },
      { name: 'RE100', code: 'Target-1', requirement: '100% renewable electricity commitment' },
      { name: 'SDG', code: '7.2', requirement: 'Renewable energy share targets' }
    ],
    factors: [],
    industry: ['All Industries', 'Manufacturing', 'Energy', 'Technology']
  },
  {
    id: 'ENERR001',
    code: 'ENERR001',
    name: 'Total Energy Consumption',
    category: 'Environmental',
    subcategory: 'Energy Management',
    description: 'Total energy consumption within the organization',
    unit: 'GJ',
    standardCodes: { GRI: ['302-1'], SASB: ['RT-EE-130a.1'], SDG: ['7.2', '7.3'] },
    dataType: 'number',
    difficulty: 'Easy',
    calculationMethod: 'Sum of all energy sources consumed including electricity, heating, cooling, and transportation',
    dataSource: ['Utility Bills', 'Energy Meters', 'Fuel Receipts', 'Energy Management Systems'],
    frequency: 'Monthly',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '302-1', requirement: 'Energy consumption within the organization' },
      { name: 'SASB', code: 'RT-EE-130a.1', requirement: 'Total energy consumed' },
      { name: 'SDG', code: '7.2', requirement: 'Monitor energy consumption patterns' }
    ],
    factors: [
      { name: 'Grid Electricity Factor', value: 3.6, unit: 'GJ/MWh', source: 'IEA' },
      { name: 'Natural Gas Factor', value: 0.0036, unit: 'GJ/m³', source: 'IPCC' }
    ],
    industry: ['All Industries', 'Manufacturing', 'Energy', 'Construction']
  },
  {
    id: 'ENERR002',
    code: 'ENERR002',
    name: 'Direct Energy Consumption',
    category: 'Environmental',
    subcategory: 'Energy Management',
    description: 'Direct energy consumption from owned or controlled sources',
    unit: 'GJ',
    standardCodes: { GRI: ['302-1'], SASB: ['RT-EE-130a.1'], SDG: ['7.2'] },
    dataType: 'number',
    difficulty: 'Easy',
    calculationMethod: 'Energy consumed from fuels and sources owned or controlled by the organization',
    dataSource: ['Fuel Purchase Records', 'On-site Generation Data', 'Fleet Fuel Records'],
    frequency: 'Monthly',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '302-1', requirement: 'Fuel consumption from non-renewable sources' },
      { name: 'SASB', code: 'RT-EE-130a.1', requirement: 'Direct energy consumption' },
      { name: 'SDG', code: '7.2', requirement: 'Track direct energy use' }
    ],
    factors: [
      { name: 'Diesel Factor', value: 0.0386, unit: 'GJ/liter', source: 'IPCC' },
      { name: 'Gasoline Factor', value: 0.0347, unit: 'GJ/liter', source: 'IPCC' }
    ],
    industry: ['Manufacturing', 'Transportation', 'Agriculture', 'Mining']
  },
  {
    id: 'ENERR005',
    code: 'ENERR005',
    name: 'Electricity',
    category: 'Environmental',
    subcategory: 'Energy Management',
    description: 'Electricity consumption',
    unit: 'kWh',
    standardCodes: { GRI: ['302-1'], SASB: ['RT-EE-130a.1'], SDG: ['7.1'] },
    dataType: 'number',
    difficulty: 'Easy',
    calculationMethod: 'Total electricity purchased and consumed from the grid',
    dataSource: ['Electricity Bills', 'Smart Meters', 'Energy Management Systems'],
    frequency: 'Monthly',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '302-1', requirement: 'Electricity consumption' },
      { name: 'SASB', code: 'RT-EE-130a.1', requirement: 'Grid electricity consumption' },
      { name: 'SDG', code: '7.1', requirement: 'Ensure access to affordable, reliable electricity' }
    ],
    factors: [
      { name: 'Grid Emission Factor', value: 0.5, unit: 'kgCO2/kWh', source: 'National Grid' }
    ],
    industry: ['All Industries', 'Manufacturing', 'Services', 'Technology']
  },

  // Water Management
  {
    id: 'WATES001',
    code: 'WATES001',
    name: 'Water Reduction Strategy',
    category: 'Environmental',
    subcategory: 'Water Management',
    description: 'Strategy for reducing water consumption',
    unit: 'Boolean',
    standardCodes: { GRI: ['303-1'], TCFD: ['Strategy-3'], SDG: ['6.4'] },
    dataType: 'boolean',
    difficulty: 'Medium',
    calculationMethod: 'Qualitative assessment of water conservation and efficiency initiatives',
    dataSource: ['Water Management Plans', 'Conservation Programs', 'Policy Documents'],
    frequency: 'Annual',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '303-1', requirement: 'Interactions with water as a shared resource' },
      { name: 'TCFD', code: 'Strategy-3', requirement: 'Water-related risks and opportunities' },
      { name: 'SDG', code: '6.4', requirement: 'Increase water-use efficiency' }
    ],
    factors: [],
    industry: ['Manufacturing', 'Agriculture', 'Food & Beverage', 'Textiles']
  },
  {
    id: 'WATER001',
    code: 'WATER001',
    name: 'Total Water Withdrawn',
    category: 'Environmental',
    subcategory: 'Water Management',
    description: 'Total water withdrawn by the organization',
    unit: 'Megalitres',
    standardCodes: { GRI: ['303-3'], SASB: ['FB-MP-140a.1'], SDG: ['6.4'] },
    dataType: 'number',
    difficulty: 'Easy',
    calculationMethod: 'Sum of all water withdrawn from surface water, groundwater, seawater, produced water, and third-party water',
    dataSource: ['Water Bills', 'Flow Meters', 'Well Records', 'Water Purchase Records'],
    frequency: 'Monthly',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '303-3', requirement: 'Water withdrawal' },
      { name: 'SASB', code: 'FB-MP-140a.1', requirement: 'Total water consumed' },
      { name: 'SDG', code: '6.4', requirement: 'Monitor water withdrawal rates' }
    ],
    factors: [
      { name: 'Water Stress Factor', value: 1.5, unit: 'multiplier', source: 'WRI Aqueduct' }
    ],
    industry: ['Manufacturing', 'Agriculture', 'Food & Beverage', 'Mining']
  },
  {
    id: 'WATER002',
    code: 'WATER002',
    name: 'Total Water Consumption',
    category: 'Environmental',
    subcategory: 'Water Management',
    description: 'Total water consumption by the organization',
    unit: 'Megalitres',
    standardCodes: { GRI: ['303-5'], SASB: ['FB-MP-140a.1'], SDG: ['6.4'] },
    dataType: 'number',
    difficulty: 'Medium',
    calculationMethod: 'Water withdrawn minus water discharged back to the same or different water body',
    dataSource: ['Water Bills', 'Flow Meters', 'Discharge Records', 'Water Balance Calculations'],
    frequency: 'Monthly',
    mandatory: true,
    standards: [
      { name: 'GRI', code: '303-5', requirement: 'Water consumption' },
      { name: 'SASB', code: 'FB-MP-140a.1', requirement: 'Total water consumed in regions with high water stress' },
      { name: 'SDG', code: '6.4', requirement: 'Monitor water consumption patterns' }
    ],
    factors: [
      { name: 'Regional Water Scarcity', value: 2.0, unit: 'multiplier', source: 'UN Water' }
    ],
    industry: ['Manufacturing', 'Agriculture', 'Food & Beverage', 'Textiles']
  }
];

// Export alias for backward compatibility
export const allSustainabilityMetrics = sustainabilityMetrics;

export const getMetricsByCategory = (category: string) => {
  return sustainabilityMetrics.filter(metric => metric.category === category);
};

export const getMetricByCode = (code: string) => {
  return sustainabilityMetrics.find(metric => metric.code === code);
};

export const getAllCategories = () => {
  return [...new Set(sustainabilityMetrics.map(metric => metric.category))];
};

export const getMetricsByStandard = (standard: string) => {
  return sustainabilityMetrics.filter(metric => 
    metric.standardCodes[standard] && metric.standardCodes[standard].length > 0
  );
};
