
export interface StorageValidationResult {
  isConfigured: boolean;
  storageType: 'none' | 'simple' | 'professional';
  provider: string;
  canWrite: boolean;
  canRead: boolean;
  errorMessage?: string;
}

export const validateStorageConfiguration = async (): Promise<StorageValidationResult> => {
  try {
    const configStr = localStorage.getItem('cloudStorageConfig');
    
    if (!configStr) {
      return {
        isConfigured: false,
        storageType: 'none',
        provider: '',
        canWrite: false,
        canRead: false,
        errorMessage: 'No storage configuration found. Please configure storage in Settings.'
      };
    }

    const config = JSON.parse(configStr);
    
    if (config.type === 'simple' && config.config?.provider === 'localstorage') {
      // Test local storage functionality
      const testKey = 'greencard_storage_test';
      const testData = { test: true, timestamp: Date.now() };
      
      try {
        localStorage.setItem(testKey, JSON.stringify(testData));
        const retrieved = localStorage.getItem(testKey);
        localStorage.removeItem(testKey);
        
        if (retrieved && JSON.parse(retrieved).test === true) {
          return {
            isConfigured: true,
            storageType: 'simple',
            provider: 'localstorage',
            canWrite: true,
            canRead: true
          };
        } else {
          throw new Error('Storage test failed');
        }
      } catch (error) {
        return {
          isConfigured: false,
          storageType: 'simple',
          provider: 'localstorage',
          canWrite: false,
          canRead: false,
          errorMessage: 'Local storage is not functioning properly. Please check browser settings.'
        };
      }
    }
    
    if (config.type === 'professional') {
      // For professional setup, verify configuration exists
      // Real validation would require backend API calls
      const hasRequiredFields = config.config?.provider && 
                                config.config?.bucketName && 
                                config.config?.accessKey;
      
      return {
        isConfigured: hasRequiredFields,
        storageType: 'professional',
        provider: config.config?.provider || '',
        canWrite: hasRequiredFields,
        canRead: hasRequiredFields,
        errorMessage: hasRequiredFields ? undefined : 'Incomplete professional storage configuration'
      };
    }
    
    return {
      isConfigured: false,
      storageType: 'none',
      provider: '',
      canWrite: false,
      canRead: false,
      errorMessage: 'Invalid storage configuration'
    };
    
  } catch (error) {
    return {
      isConfigured: false,
      storageType: 'none',
      provider: '',
      canWrite: false,
      canRead: false,
      errorMessage: 'Error validating storage configuration'
    };
  }
};

export const requireStorageConfiguration = async (): Promise<void> => {
  const validation = await validateStorageConfiguration();
  
  if (!validation.isConfigured || !validation.canWrite) {
    throw new Error(
      validation.errorMessage || 
      'Storage must be configured before you can save data. Please go to Settings > Data Storage Configuration.'
    );
  }
};

export const getStorageInfo = async () => {
  const validation = await validateStorageConfiguration();
  return validation;
};
