
-- Create user roles table
CREATE TYPE public.user_role AS ENUM ('admin', 'data_entry', 'analyst', 'executive');

CREATE TABLE public.user_profiles (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE,
  role user_role NOT NULL DEFAULT 'data_entry',
  company_name TEXT,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Create ESG data categories table
CREATE TABLE public.esg_data_categories (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  name TEXT NOT NULL,
  category_type TEXT NOT NULL CHECK (category_type IN ('Environmental', 'Social', 'Governance', 'Economic')),
  description TEXT,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Create main ESG data points table
CREATE TABLE public.esg_data_points (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE,
  category_id UUID REFERENCES public.esg_data_categories(id),
  metric_name TEXT NOT NULL,
  metric_value DECIMAL,
  unit TEXT,
  reporting_period DATE,
  scope TEXT, -- For GHG emissions (scope 1, 2, 3)
  data_source TEXT,
  validation_status TEXT DEFAULT 'pending' CHECK (validation_status IN ('pending', 'validated', 'flagged')),
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW(),
  updated_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Create reports table
CREATE TABLE public.sustainability_reports (
  id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
  user_id UUID REFERENCES auth.users(id) ON DELETE CASCADE,
  report_name TEXT NOT NULL,
  standards TEXT[], -- Array of standards like ['IFRS-S1', 'GRI', 'TCFD']
  report_type TEXT,
  status TEXT DEFAULT 'draft' CHECK (status IN ('draft', 'published', 'archived')),
  configuration JSONB, -- Store report customization settings
  generated_at TIMESTAMP WITH TIME ZONE,
  created_at TIMESTAMP WITH TIME ZONE DEFAULT NOW()
);

-- Enable RLS on all tables
ALTER TABLE public.user_profiles ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.esg_data_categories ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.esg_data_points ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.sustainability_reports ENABLE ROW LEVEL SECURITY;

-- RLS Policies for user_profiles
CREATE POLICY "Users can view own profile" ON public.user_profiles
  FOR SELECT USING (auth.uid() = user_id);

CREATE POLICY "Users can update own profile" ON public.user_profiles
  FOR UPDATE USING (auth.uid() = user_id);

CREATE POLICY "Users can insert own profile" ON public.user_profiles
  FOR INSERT WITH CHECK (auth.uid() = user_id);

-- RLS Policies for esg_data_categories (public read)
CREATE POLICY "Anyone can view categories" ON public.esg_data_categories
  FOR SELECT TO authenticated USING (true);

-- RLS Policies for esg_data_points
CREATE POLICY "Users can view own data" ON public.esg_data_points
  FOR SELECT USING (auth.uid() = user_id);

CREATE POLICY "Users can insert own data" ON public.esg_data_points
  FOR INSERT WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update own data" ON public.esg_data_points
  FOR UPDATE USING (auth.uid() = user_id);

CREATE POLICY "Users can delete own data" ON public.esg_data_points
  FOR DELETE USING (auth.uid() = user_id);

-- RLS Policies for sustainability_reports
CREATE POLICY "Users can view own reports" ON public.sustainability_reports
  FOR SELECT USING (auth.uid() = user_id);

CREATE POLICY "Users can insert own reports" ON public.sustainability_reports
  FOR INSERT WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update own reports" ON public.sustainability_reports
  FOR UPDATE USING (auth.uid() = user_id);

CREATE POLICY "Users can delete own reports" ON public.sustainability_reports
  FOR DELETE USING (auth.uid() = user_id);

-- Insert sample ESG data categories
INSERT INTO public.esg_data_categories (name, category_type, description) VALUES
('Energy Consumption', 'Environmental', 'Total energy consumed across operations'),
('GHG Emissions Scope 1', 'Environmental', 'Direct greenhouse gas emissions from owned sources'),
('GHG Emissions Scope 2', 'Environmental', 'Indirect emissions from purchased energy'),
('GHG Emissions Scope 3', 'Environmental', 'Indirect emissions from value chain'),
('Water Usage', 'Environmental', 'Total water consumption'),
('Waste Generation', 'Environmental', 'Total waste produced and disposal methods'),
('Employee Diversity', 'Social', 'Workforce diversity metrics'),
('Health & Safety', 'Social', 'Workplace safety incidents and programs'),
('Board Composition', 'Governance', 'Board diversity and independence metrics'),
('Ethics & Compliance', 'Governance', 'Compliance training and ethics violations'),
('Revenue', 'Economic', 'Total revenue generated'),
('R&D Investment', 'Economic', 'Investment in research and development');
