
-- Create audit_logs table for tracking user actions
CREATE TABLE public.audit_logs (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID REFERENCES auth.users NOT NULL,
  action TEXT NOT NULL,
  details JSONB,
  timestamp TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  user_email TEXT
);

-- Enable RLS on audit_logs
ALTER TABLE public.audit_logs ENABLE ROW LEVEL SECURITY;

-- Create policies for audit_logs
CREATE POLICY "Users can view own audit logs" 
  ON public.audit_logs 
  FOR SELECT 
  USING (auth.uid() = user_id);

CREATE POLICY "Users can insert own audit logs" 
  ON public.audit_logs 
  FOR INSERT 
  WITH CHECK (auth.uid() = user_id);

-- Create team_members table for organization team management
CREATE TABLE public.team_members (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  organization_id UUID REFERENCES auth.users NOT NULL,
  full_name TEXT NOT NULL,
  email TEXT NOT NULL,
  role TEXT NOT NULL DEFAULT 'sustainability_officer',
  profile_picture TEXT,
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

-- Enable RLS on team_members
ALTER TABLE public.team_members ENABLE ROW LEVEL SECURITY;

-- Create policies for team_members
CREATE POLICY "Users can view own team members" 
  ON public.team_members 
  FOR SELECT 
  USING (auth.uid() = organization_id);

CREATE POLICY "Users can insert own team members" 
  ON public.team_members 
  FOR INSERT 
  WITH CHECK (auth.uid() = organization_id);

CREATE POLICY "Users can update own team members" 
  ON public.team_members 
  FOR UPDATE 
  USING (auth.uid() = organization_id);

CREATE POLICY "Users can delete own team members" 
  ON public.team_members 
  FOR DELETE 
  USING (auth.uid() = organization_id);

-- Create report_templates table for ESG reporting templates
CREATE TABLE public.report_templates (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  name TEXT NOT NULL,
  description TEXT,
  template_type TEXT NOT NULL,
  template_data JSONB NOT NULL,
  is_public BOOLEAN DEFAULT true,
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

-- Enable RLS on report_templates
ALTER TABLE public.report_templates ENABLE ROW LEVEL SECURITY;

-- Create policy for report_templates (read-only for all authenticated users)
CREATE POLICY "All users can view report templates" 
  ON public.report_templates 
  FOR SELECT 
  TO authenticated
  USING (true);

-- Create user_report_customizations table for storing user customizations
CREATE TABLE public.user_report_customizations (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID REFERENCES auth.users NOT NULL,
  template_id UUID REFERENCES public.report_templates NOT NULL,
  customizations JSONB NOT NULL,
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  UNIQUE(user_id, template_id)
);

-- Enable RLS on user_report_customizations
ALTER TABLE public.user_report_customizations ENABLE ROW LEVEL SECURITY;

-- Create policies for user_report_customizations
CREATE POLICY "Users can view own customizations" 
  ON public.user_report_customizations 
  FOR SELECT 
  USING (auth.uid() = user_id);

CREATE POLICY "Users can insert own customizations" 
  ON public.user_report_customizations 
  FOR INSERT 
  WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update own customizations" 
  ON public.user_report_customizations 
  FOR UPDATE 
  USING (auth.uid() = user_id);

-- Insert sample report templates
INSERT INTO public.report_templates (name, description, template_type, template_data) VALUES
('Annual Sustainability Report', 'Comprehensive annual sustainability report covering ESG metrics', 'annual', '{"sections": ["executive_summary", "environmental_metrics", "social_metrics", "governance_metrics", "goals_targets"], "format": "pdf", "pages": 25}'),
('TCFD Climate Report', 'Task Force on Climate-related Financial Disclosures report', 'climate', '{"sections": ["governance", "strategy", "risk_management", "metrics_targets"], "format": "pdf", "pages": 15}'),
('GRI Standards Report', 'Global Reporting Initiative standards compliance report', 'gri', '{"sections": ["organizational_profile", "strategy", "ethics", "governance", "stakeholder_engagement"], "format": "pdf", "pages": 30}'),
('ESG Performance Dashboard', 'Executive dashboard for ESG key performance indicators', 'dashboard', '{"sections": ["kpi_overview", "environmental_charts", "social_charts", "governance_charts"], "format": "html", "interactive": true}'),
('Carbon Footprint Summary', 'Detailed carbon footprint analysis and reduction strategies', 'carbon', '{"sections": ["scope1_emissions", "scope2_emissions", "scope3_emissions", "reduction_targets"], "format": "pdf", "pages": 12}'),
('Quarterly ESG Update', 'Quarterly progress report on sustainability initiatives', 'quarterly', '{"sections": ["progress_summary", "key_achievements", "challenges", "next_quarter_goals"], "format": "pdf", "pages": 8}');
