
-- Add new tables for enhanced functionality

-- Carbon accounting calculations with currency support
CREATE TABLE IF NOT EXISTS public.carbon_calculations (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID REFERENCES auth.users NOT NULL,
  calculation_type TEXT NOT NULL, -- 'energy', 'waste', 'plastic', 'chemical', 'production'
  scope TEXT NOT NULL, -- 'scope-1', 'scope-2', 'scope-3'
  activity_name TEXT NOT NULL,
  amount DECIMAL NOT NULL,
  unit TEXT NOT NULL,
  emission_factor DECIMAL NOT NULL,
  co2_equivalent DECIMAL NOT NULL,
  local_currency TEXT DEFAULT 'USD',
  local_amount DECIMAL,
  usd_amount DECIMAL,
  exchange_rate DECIMAL DEFAULT 1.0,
  calculation_date DATE NOT NULL DEFAULT CURRENT_DATE,
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

-- Risk management assessments
CREATE TABLE IF NOT EXISTS public.risk_assessments (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID REFERENCES auth.users NOT NULL,
  risk_category TEXT NOT NULL, -- 'climate', 'regulatory', 'operational', 'financial', 'reputational'
  risk_name TEXT NOT NULL,
  risk_description TEXT,
  likelihood INTEGER CHECK (likelihood >= 1 AND likelihood <= 5), -- 1-5 scale
  impact INTEGER CHECK (impact >= 1 AND impact <= 5), -- 1-5 scale
  risk_score INTEGER GENERATED ALWAYS AS (likelihood * impact) STORED,
  mitigation_strategy TEXT,
  responsible_person TEXT,
  review_date DATE,
  status TEXT DEFAULT 'active', -- 'active', 'mitigated', 'closed'
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

-- Currency exchange rates table
CREATE TABLE IF NOT EXISTS public.currency_rates (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  currency_code TEXT NOT NULL,
  usd_rate DECIMAL NOT NULL,
  country TEXT,
  region TEXT,
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  UNIQUE(currency_code)
);

-- Insert some common currency rates
INSERT INTO public.currency_rates (currency_code, usd_rate, country, region) VALUES
('USD', 1.0, 'United States', 'North America'),
('EUR', 1.08, 'European Union', 'Europe'),
('GBP', 1.27, 'United Kingdom', 'Europe'),
('INR', 0.012, 'India', 'Asia'),
('CAD', 0.74, 'Canada', 'North America'),
('AUD', 0.66, 'Australia', 'Oceania'),
('JPY', 0.0067, 'Japan', 'Asia'),
('CNY', 0.14, 'China', 'Asia')
ON CONFLICT (currency_code) DO NOTHING;

-- Add RLS policies for carbon_calculations
ALTER TABLE public.carbon_calculations ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Users can view their own carbon calculations" 
  ON public.carbon_calculations 
  FOR SELECT 
  USING (auth.uid() = user_id);

CREATE POLICY "Users can create their own carbon calculations" 
  ON public.carbon_calculations 
  FOR INSERT 
  WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update their own carbon calculations" 
  ON public.carbon_calculations 
  FOR UPDATE 
  USING (auth.uid() = user_id);

CREATE POLICY "Users can delete their own carbon calculations" 
  ON public.carbon_calculations 
  FOR DELETE 
  USING (auth.uid() = user_id);

-- Add RLS policies for risk_assessments
ALTER TABLE public.risk_assessments ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Users can view their own risk assessments" 
  ON public.risk_assessments 
  FOR SELECT 
  USING (auth.uid() = user_id);

CREATE POLICY "Users can create their own risk assessments" 
  ON public.risk_assessments 
  FOR INSERT 
  WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update their own risk assessments" 
  ON public.risk_assessments 
  FOR UPDATE 
  USING (auth.uid() = user_id);

CREATE POLICY "Users can delete their own risk assessments" 
  ON public.risk_assessments 
  FOR DELETE 
  USING (auth.uid() = user_id);

-- Allow everyone to read currency rates
ALTER TABLE public.currency_rates ENABLE ROW LEVEL SECURITY;

CREATE POLICY "Everyone can view currency rates" 
  ON public.currency_rates 
  FOR SELECT 
  USING (true);
