
-- Portfolio Management tables
CREATE TABLE IF NOT EXISTS public.portfolio_investments (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID NOT NULL,
  investment_name TEXT NOT NULL,
  investment_type TEXT NOT NULL, -- 'equity', 'bond', 'fund', 'real_estate', 'alternative'
  sector TEXT,
  region TEXT,
  investment_amount DECIMAL NOT NULL,
  currency TEXT DEFAULT 'USD',
  esg_score DECIMAL CHECK (esg_score >= 0 AND esg_score <= 100),
  environmental_score DECIMAL CHECK (environmental_score >= 0 AND environmental_score <= 100),
  social_score DECIMAL CHECK (social_score >= 0 AND social_score <= 100),
  governance_score DECIMAL CHECK (governance_score >= 0 AND governance_score <= 100),
  carbon_intensity DECIMAL, -- tCO2e per million revenue
  water_intensity DECIMAL,
  waste_intensity DECIMAL,
  investment_date DATE NOT NULL,
  last_reviewed DATE,
  status TEXT DEFAULT 'active', -- 'active', 'divested', 'under_review'
  notes TEXT,
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

-- Net-Zero Pathway tables
CREATE TABLE IF NOT EXISTS public.net_zero_targets (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID NOT NULL,
  target_name TEXT NOT NULL,
  target_type TEXT NOT NULL, -- 'absolute', 'intensity', 'renewable_energy', 'carbon_neutral'
  scope TEXT NOT NULL, -- 'scope-1', 'scope-2', 'scope-3', 'all-scopes'
  baseline_year INTEGER NOT NULL,
  baseline_emissions DECIMAL NOT NULL,
  target_year INTEGER NOT NULL,
  target_emissions DECIMAL NOT NULL,
  reduction_percentage DECIMAL GENERATED ALWAYS AS (
    CASE 
      WHEN baseline_emissions > 0 THEN ((baseline_emissions - target_emissions) / baseline_emissions * 100)
      ELSE 0
    END
  ) STORED,
  current_emissions DECIMAL,
  progress_percentage DECIMAL GENERATED ALWAYS AS (
    CASE 
      WHEN baseline_emissions > 0 AND current_emissions IS NOT NULL THEN 
        ((baseline_emissions - current_emissions) / baseline_emissions * 100)
      ELSE 0
    END
  ) STORED,
  status TEXT DEFAULT 'active', -- 'active', 'achieved', 'at_risk', 'failed'
  methodology TEXT, -- 'SBTi', 'Paris_Agreement', 'Custom'
  verification_status TEXT DEFAULT 'unverified', -- 'verified', 'unverified', 'pending'
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.decarbonization_initiatives (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID NOT NULL,
  target_id UUID REFERENCES public.net_zero_targets(id) ON DELETE CASCADE,
  initiative_name TEXT NOT NULL,
  initiative_type TEXT NOT NULL, -- 'energy_efficiency', 'renewable_energy', 'process_improvement', 'fuel_switching', 'carbon_capture', 'nature_based'
  description TEXT,
  estimated_reduction DECIMAL NOT NULL, -- tCO2e per year
  investment_required DECIMAL,
  currency TEXT DEFAULT 'USD',
  implementation_date DATE,
  completion_date DATE,
  status TEXT DEFAULT 'planned', -- 'planned', 'in_progress', 'completed', 'cancelled'
  actual_reduction DECIMAL,
  roi_percentage DECIMAL,
  responsible_team TEXT,
  priority_level TEXT DEFAULT 'medium', -- 'high', 'medium', 'low'
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

-- Supply Chain tables
CREATE TABLE IF NOT EXISTS public.suppliers (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID NOT NULL,
  supplier_name TEXT NOT NULL,
  supplier_type TEXT NOT NULL, -- 'tier_1', 'tier_2', 'tier_3', 'critical', 'non_critical'
  industry_sector TEXT,
  country TEXT,
  region TEXT,
  contact_email TEXT,
  contact_phone TEXT,
  annual_spend DECIMAL,
  currency TEXT DEFAULT 'USD',
  esg_assessment_score DECIMAL CHECK (esg_assessment_score >= 0 AND esg_assessment_score <= 100),
  carbon_footprint DECIMAL, -- tCO2e annually
  water_usage DECIMAL,
  waste_generation DECIMAL,
  labor_practices_score DECIMAL CHECK (labor_practices_score >= 0 AND labor_practices_score <= 100),
  environmental_compliance BOOLEAN DEFAULT false,
  social_compliance BOOLEAN DEFAULT false,
  governance_compliance BOOLEAN DEFAULT false,
  certification_standards TEXT[], -- array of certifications like ISO14001, SA8000, etc
  risk_level TEXT DEFAULT 'medium', -- 'high', 'medium', 'low'
  last_assessment_date DATE,
  next_assessment_date DATE,
  status TEXT DEFAULT 'active', -- 'active', 'under_review', 'suspended', 'terminated'
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

CREATE TABLE IF NOT EXISTS public.supply_chain_assessments (
  id UUID NOT NULL DEFAULT gen_random_uuid() PRIMARY KEY,
  user_id UUID NOT NULL,
  supplier_id UUID REFERENCES public.suppliers(id) ON DELETE CASCADE,
  assessment_type TEXT NOT NULL, -- 'initial', 'annual', 'audit', 'incident_based'
  assessment_date DATE NOT NULL,
  assessor_name TEXT,
  overall_score DECIMAL CHECK (overall_score >= 0 AND overall_score <= 100),
  environmental_score DECIMAL CHECK (environmental_score >= 0 AND environmental_score <= 100),
  social_score DECIMAL CHECK (social_score >= 0 AND social_score <= 100),
  governance_score DECIMAL CHECK (governance_score >= 0 AND governance_score <= 100),
  findings TEXT,
  recommendations TEXT,
  action_items TEXT,
  follow_up_date DATE,
  status TEXT DEFAULT 'pending', -- 'completed', 'pending', 'in_progress'
  created_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITH TIME ZONE NOT NULL DEFAULT now()
);

-- Add RLS policies for all new tables
ALTER TABLE public.portfolio_investments ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.net_zero_targets ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.decarbonization_initiatives ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.suppliers ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.supply_chain_assessments ENABLE ROW LEVEL SECURITY;

-- Portfolio investments policies
CREATE POLICY "Users can manage their own portfolio investments" ON public.portfolio_investments
  FOR ALL USING (auth.uid() = user_id) WITH CHECK (auth.uid() = user_id);

-- Net-zero targets policies  
CREATE POLICY "Users can manage their own net-zero targets" ON public.net_zero_targets
  FOR ALL USING (auth.uid() = user_id) WITH CHECK (auth.uid() = user_id);

-- Decarbonization initiatives policies
CREATE POLICY "Users can manage their own decarbonization initiatives" ON public.decarbonization_initiatives
  FOR ALL USING (auth.uid() = user_id) WITH CHECK (auth.uid() = user_id);

-- Suppliers policies
CREATE POLICY "Users can manage their own suppliers" ON public.suppliers
  FOR ALL USING (auth.uid() = user_id) WITH CHECK (auth.uid() = user_id);

-- Supply chain assessments policies
CREATE POLICY "Users can manage their own supply chain assessments" ON public.supply_chain_assessments
  FOR ALL USING (auth.uid() = user_id) WITH CHECK (auth.uid() = user_id);
